<?php
// CARGA MAESTRA DE CONFIGURACIÓN Y BRANDING
require_once(__DIR__ . '/core/init.php');

$servicios_data_from_db = [];
$monthly_sold_services_count = 0;
$total_active_clients_count = 0;

try {

    $sql_services = "
        SELECT 
            s.id, 
            s.name, 
            s.price, 
            s.category, 
            s.description,
            s.image_path, 
            s.features,
            s.youtube_link,
            s.gallery_images,
            COUNT(cs.client_id) as active_clients
        FROM services s
        LEFT JOIN client_services cs ON s.id = cs.service_id
        GROUP BY s.id
        ORDER BY s.name ASC
    ";
    $stmt_services = $pdo->prepare($sql_services);
    $stmt_services->execute();
    
    while ($row = $stmt_services->fetch(PDO::FETCH_ASSOC)) {
        $key = 'service-' . $row['id'];
        
        $servicios_data_from_db[$key] = [
            'id' => $key,
            'db_id' => (int)$row['id'],
            'nombre' => htmlspecialchars($row['name']),
            'precio' => (float)$row['price'],
            'categoria' => htmlspecialchars($row['category'] ?? ''),
            'descripcion' => $row['description'] ?? '', 
            'features' => $row['features'] ?? '',       
            'imagen' => htmlspecialchars($row['image_path'] ?? ''), 
            'youtube_link' => htmlspecialchars($row['youtube_link'] ?? ''), 
            'gallery_images' => htmlspecialchars($row['gallery_images'] ?? ''), 
            'active_clients' => (int)$row['active_clients']
        ];
    }
    
    // 2. Total de Clientes Activos
    $stmt_active_clients = $pdo->prepare("SELECT COUNT(DISTINCT client_id) FROM client_services");
    $stmt_active_clients->execute();
    $total_active_clients_count = (int)$stmt_active_clients->fetchColumn();

    // 3. Servicios Vendidos (Mes Actual)
    $current_month = date('m');
    $current_year = date('Y');

    $stmt_sold = $pdo->prepare("
        SELECT SUM(ii.quantity) 
        FROM invoice_items ii
        JOIN invoices i ON ii.invoice_id = i.id
        WHERE i.status = 'pagada' 
        AND MONTH(i.invoice_date) = :m AND YEAR(i.invoice_date) = :y
    ");
    $stmt_sold->execute([':m' => $current_month, ':y' => $current_year]);
    $monthly_sold_services_count = (int)$stmt_sold->fetchColumn();

} catch (PDOException $e) {
    error_log("Error en services.php: " . $e->getMessage());
} catch (Exception $e) {
    error_log("Error general en services.php: " . $e->getMessage());
}

$servicios_json = json_encode($servicios_data_from_db, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP);
$monthly_sold_services_json = json_encode($monthly_sold_services_count);

if (ob_get_level() > 0) {
    ob_end_clean();
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Catálogo De Productos Y Servicios <?php echo htmlspecialchars($branding['full_title']); ?></title>
    <meta name="robots" content="noindex, nofollow">
    
    <link rel="icon" type="image/png" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    <link rel="apple-touch-icon" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="<?php echo htmlspecialchars($google_font_url); ?>">

    <?php include 'files/gtm-head.php'; ?>
    
    <script src="https://unpkg.com/lucide@latest"></script>
    <script src="https://unpkg.com/imask"></script>
    
    <link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
    <script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>

    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script>
    
    <script src="files/editor.js"></script>
</head>

<body data-page-title="Catálogo De Productos Y Servicios "
      data-page-subtitle="Gestión De Propuestas Comerciales Para El Mercado"
      data-page-icon="briefcase-business">

    <div id="toast-container" class="toast-container"></div>
    
<?php include 'files/gtm-body.php'; ?> 

<div class="relative min-h-screen md:flex">
    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>
    <div id="task-panel-overlay" class="fixed inset-0 bg-black/60 z-40 hidden transition-opacity duration-300"></div>  
    
    <?php include 'menu.php'; ?>
    
    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>

        <div id="content-area" class="p-4 md:p-8 space-y-8">
            
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
                    <i data-lucide="package" class="w-12 h-12 text-[var(--color-secondary)]"></i>
                    <div>
                        <h3 class="text-lg font-black text-gray-500 mb-1">TOTAL DE SERVICIOS</h3>
                        <p id="total-services-count" class="text-5xl font-bold text-[var(--color-primary)]">0</p>
                    </div>
                </div>

                <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
                    <i data-lucide="users" class="w-12 h-12 text-[var(--color-secondary)]"></i>
                    <div>
                        <h3 class="text-lg font-black text-gray-500 mb-1">CLIENTES ACTIVOS</h3>
                        <p id="total-active-clients-count" class="text-5xl font-bold text-[var(--color-primary)]"><?php echo $total_active_clients_count; ?></p>
                    </div>
                </div>

                <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
                    <i data-lucide="wallet" class="w-12 h-12 text-[var(--color-secondary)]"></i>
                    <div>
                        <h3 class="text-lg font-black text-gray-500 mb-1">SERVICIOS VENDIDOS</h3>
                        <p id="monthly-sold-services-count" class="text-5xl font-bold text-[var(--color-primary)]">0</p>
                    </div>
                </div>
            </div>
            
            <section id="servicios" class="dashboard-section">
                <div class="bg-white p-6 rounded-xl shadow-md">
                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4">
                        <div>
                            <h3 class="text-2xl font-extrabold text-gray-800 flex items-center gap-2">
                                <i data-lucide="package" class="w-7 h-7 text-[var(--color-primary)]"></i> INVENTARIO DE SERVICIOS
                            </h3>
                            <p class="text-gray-500 text-sm mt-1 uppercase">VISUALIZA Y EDITA TUS PRODUCTOS Y SERVICIOS.</p>
                        </div>
                        <div class="flex flex-col sm:flex-row items-center gap-4 w-full md:w-auto">
                            <button class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2" onclick="openPanel('addServicioPanel')">
                                <i data-lucide="plus" class="w-5 h-5 mr-2"></i> AGREGAR SERVICIO
                            </button>
                            <button class="w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2" onclick="openPanel('manageCategoriesPanel')">
                                <i data-lucide="tag" class="w-4 h-4 mr-2"></i> ADMINISTRAR CATEGORÍAS
                            </button>
                        </div>
                    </div>

                    <div class="flex flex-col md:flex-row gap-4 mb-6">
                        <div class="relative flex-grow">
                            <input type="text" id="service-search" placeholder="Buscar Por Nombre o Descripción..." class="w-full p-3 pl-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]">
                            <i data-lucide="search" class="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"></i>
                        </div>
                        <select id="service-filter-category" class="w-full md:w-48 p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white uppercase text-sm font-bold text-gray-700">
                            <option value="all">TODAS LAS CATEGORÍAS</option>
                        </select>
                        <div class="view-toggle inline-flex bg-gray-200 rounded-lg p-1 shrink-0">
                            <button id="grid-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 active uppercase bg-white shadow-sm text-gray-800" title="Vista de Cuadrícula">
                                <i data-lucide="layout-grid" class="w-5 h-5"></i>
                            </button>
                            <button id="list-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 uppercase text-gray-500 hover:bg-white/50" title="Vista de Lista">
                                <i data-lucide="list" class="w-5 h-5"></i>
                            </button>
                        </div>
                    </div>

                    <div id="service-grid-view" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6"></div>

                    <div id="service-list-view" class="overflow-x-auto hidden">
                        <table class="min-w-full bg-white text-sm text-left">
                            <thead class="bg-gray-50 hidden md:table-header-group">
                                <tr class="text-left text-gray-500 uppercase text-sm border-b border-gray-200">
                                    <th class="py-3 px-6 font-semibold">NOMBRE</th>
                                    <th class="py-3 px-6 font-semibold hidden md:table-cell">CATEGORÍA</th>
                                    <th class="py-3 px-6 font-semibold">PRECIO</th>
                                    <th class="py-3 px-6 font-semibold text-center">CLIENTES</th>
                                    <th class="py-3 px-6 font-semibold text-center">ACCIONES</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-700 text-sm" id="servicesTableBody">
                            </tbody>
                        </table>
                    </div>

                </div>
            </section>
        </div>
    </main>
</div>

<div id="addServicioPanel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="plus-circle" class="w-8 h-8 mr-2 text-white"></i>
                AGREGAR SERVICIO
            </h3>
            <button onclick="closePanel('addServicioPanel')" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
                <i data-lucide="x" class="w-8 h-8"></i>
            </button>
        </div>

        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <div class="border-b pb-3 mb-4">
                    <h4 class="text-2xl font-black text-[var(--color-secondary)] uppercase flex items-center">
                        <i data-lucide="clipboard-list" class="w-5 h-5 mr-2"></i>
                        DETALLES DEL SERVICIO
                    </h4>
                </div>

                <form id="add-servicio-form" class="space-y-4">
                    
                    <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg space-y-4">
                                            <div>
                        <label for="servicio-nombre" class="block text-md font-black text-black mb-1 uppercase">Nombre del Servicio</label>
                        <input type="text" id="servicio-nombre" name="name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Ej. Limpieza Semanal" required>
                    </div>
                        <div class="border-b pb-2 mb-2">
                            <label class="block text-md font-black text-black mb-1 uppercase">PORTADA (Imagen o Video)</label>
                        </div>

                        <div>
                            <input type="url" id="servicio-youtube-link" name="youtube_link" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="https://www.youtube.com/watch?v=...">
                        </div>

                        <div class="py-2 border-t border-gray-200">
                            <label for="servicio-imagen" class="block text-sm font-medium text-gray-600 mb-1">Subir Imagen de Portada</label>
                            <input type="file" id="servicio-imagen" name="image" accept="image/*" class="w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-bold file:bg-[var(--color-primary)] file:text-white hover:file:bg-gray-700">
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="servicio-precio" class="block text-md font-black text-black mb-1 uppercase">Precio</label>
                            <input type="number" step="0.01" id="servicio-precio" name="price" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Ej. 50.00" required>
                        </div>
                        <div>
                            <label for="servicio-categoria" class="block text-md font-black text-black mb-1 uppercase">Categoría</label>
                            <select id="servicio-categoria" name="category" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm bg-white">
                                <option value="">Selecciona Una Categoría</option>
                            </select>
                        </div>
                    </div>

                    <div>
                        <label class="block text-md font-black text-black mb-1 uppercase">Características</label>
                        <input type="hidden" id="add-features-hidden" name="features">
                        <div id="add-features-quill" class="min-h-[100px] border border-gray-300 rounded-lg bg-white"></div>
                        <p class="text-xs text-gray-400 mt-1">Utiliza listas y formato para detallar los puntos clave.</p>
                    </div>
                    <div>
                        <label class="block text-md font-black text-black mb-1 uppercase">Descripción</label>
                        <input type="hidden" id="add-description-hidden" name="description">
                        <div id="add-description-quill" class="min-h-[100px] border border-gray-300 rounded-lg bg-white"></div>
                    </div>
                    <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg space-y-4">
                        <div class="border-b pb-2 mb-2">
                            <label class="block text-md font-black text-black mb-1 uppercase">GALERÍA DE FOTOS (Máx. 4)</label>
                            <p class="text-xs text-gray-400 mt-1 uppercase">Selecciona hasta 4 imágenes para la galería del servicio.</p>
                        </div>
                        <input type="file" id="servicio-galeria-imagenes" name="gallery_images[]" accept="image/*" multiple class="w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-bold file:bg-[var(--color-secondary)] file:text-white hover:file:bg-gray-700">
                    </div>
                    </form>
            </div>
        </div>

        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <div class="grid grid-cols-2 gap-3">
                <button type="button" class="w-full bg-[var(--color-primary)] hover:bg-gray-800 text-white font-bold py-2 px-4 rounded-lg uppercase transition-all flex items-center justify-center gap-2 shadow-sm" onclick="closePanel('addServicioPanel')">
                    <i data-lucide="x-circle" class="w-5 h-5"></i> CANCELAR
                </button>
                <button type="submit" form="add-servicio-form" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-bold py-2 px-4 rounded-lg uppercase transition-all flex items-center justify-center gap-2 shadow-md">
                    <i data-lucide="save" class="w-5 h-5"></i> GUARDAR
                </button>
            </div>
        </div>
    </div>
</div>

<div id="viewServicioPanel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="file-edit" class="w-8 h-8 mr-2 text-white"></i>
                DETALLE SERVICIO
            </h3>
            <button onclick="closePanel('viewServicioPanel')" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
                <i data-lucide="x" class="w-8 h-8"></i>
            </button>
        </div>

        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <div class="border-b pb-3 mb-4">
                    <h4 class="text-2xl font-black text-[var(--color-secondary)] uppercase flex items-center">
                        <i data-lucide="edit" class="w-5 h-5 mr-2"></i>
                        MODIFICAR SERVICIO
                    </h4>
                </div>
                <form id="view-servicio-form" class="space-y-4">
                    <input type="hidden" id="view-servicio-id" name="id">
                    <input type="hidden" id="view-current-image-path" name="current_image_path">
                    <input type="hidden" id="view-current-gallery-images" name="current_gallery_images">
                    
                    <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg space-y-4">
                        <div class="border-b pb-2 mb-2">
                            <h5 class="text-md font-black text-black mb-1 uppercase">PORTADA ACTUAL</h5>
                            <div id="view-servicio-preview" class="h-40 bg-gray-200 flex items-center justify-center rounded-lg border border-dashed border-gray-300">
                                <i data-lucide="image" class="w-8 h-8 text-gray-400"></i>
                            </div>
                        </div>
                    <div>
                        <label for="view-servicio-nombre" class="block text-md font-black text-black mb-1 uppercase">Nombre del Servicio</label>
                        <input type="text" id="view-servicio-nombre" name="name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Nombre Del servicio">
                    </div>
                        <div class="border-b pb-2 mb-2">
                            <label class="block text-md font-black text-black mb-1 uppercase">CAMBIAR PORTADA (Imagen o Video)</label>
                        </div>

                        <div>
                            <label for="view-servicio-youtube-link" class="block text-sm font-medium text-gray-600 mb-1 uppercase">URL de Video de YouTube (Opcional)</label>
                            <input type="url" id="view-servicio-youtube-link" name="youtube_link" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="https://www.youtube.com/watch?v=...">
                        </div>

                        <div class="py-2 border-t border-gray-200">
                            <label for="view-servicio-imagen" class="block text-sm font-medium text-gray-600 mb-1">Subir Nueva Imagen de Portada</label>
                            <input type="file" id="view-servicio-imagen" name="image" accept="image/*" class="w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-bold file:bg-[var(--color-primary)] file:text-white hover:file:bg-gray-700">
                        </div>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="view-servicio-precio" class="block text-md font-black text-black mb-1 uppercase">Precio</label>
                            <input type="number" step="0.01" id="view-servicio-precio" name="price" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Ej. 50.00">
                        </div>
                        <div>
                            <label for="view-servicio-categoria" class="block text-md font-black text-black mb-1 uppercase">Categoría</label>
                            <select id="view-servicio-categoria" name="category" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm bg-white">
                                <option value="">Selecciona Una Categoría</option>
                            </select>
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-md font-black text-black mb-1 uppercase">Características</label>
                        <input type="hidden" id="view-features-hidden" name="features">
                        <div id="view-features-quill" class="min-h-[100px] border border-gray-300 rounded-lg bg-white"></div>
                    </div>
                    <div>
                        <label class="block text-md font-black text-black mb-1 uppercase">Descripción</label>
                        <input type="hidden" id="view-description-hidden" name="description">
                        <div id="view-description-quill" class="min-h-[100px] border border-gray-300 rounded-lg bg-white"></div>
                    </div>
                    <div class="bg-gray-50 p-4 border border-gray-200 rounded-lg space-y-4">
                        <div class="border-b pb-2 mb-2">
                            <label class="block text-md font-black text-black mb-1 uppercase">GALERÍA DE FOTOS (Máx. 4)</label>
                            <p class="text-xs text-[var(--color-secondary)] font-bold mt-1">ADVERTENCIA: Si subes nuevas fotos, la galería anterior será reemplazada.</p>
                        </div>

                        <div id="view-servicio-gallery-preview" class="flex gap-2 mb-3 overflow-x-auto p-1">
                            <p class="text-xs text-gray-400 italic">No hay fotos en la galería actual.</p>
                        </div>

                        <label for="view-servicio-galeria-imagenes" class="block text-sm font-medium text-gray-600 mb-1">Subir Nuevas Fotos de Galería</label>
                        <input type="file" id="view-servicio-galeria-imagenes" name="gallery_images[]" accept="image/*" multiple class="w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-bold file:bg-[var(--color-secondary)] file:text-white hover:file:bg-gray-700">
                        <div id="view-gallery-counter" class="text-xs text-gray-400 mt-1"></div>
                    </div>
                    </form>
            </div>
        </div>

        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <div class="grid grid-cols-2 gap-3">
                <button type="button" class="w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2" onclick="closePanel('viewServicioPanel')">
                    <i data-lucide="x-circle" class="w-5 h-5"></i> CANCELAR
                </button>
                <button type="submit" form="view-servicio-form" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2">
                    <i data-lucide="save" class="w-5 h-5"></i> GUARDAR CAMBIOS
                </button>
            </div>
        </div>
    </div>
</div>

<div id="viewClientsPanel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 id="clients-panel-title" class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="user" class="w-8 h-8 mr-2 text-white"></i>
                CLIENTES ASIGNADOS
            </h3>
            <button onclick="closePanel('viewClientsPanel')" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
                <i data-lucide="x" class="w-8 h-8"></i>
            </button>
        </div>

        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <div class="border-b pb-3 mb-4">
                    <h4 class="text-2xl font-black text-[var(--color-secondary)] uppercase flex items-center">
                        <i data-lucide="list" class="w-5 h-5 mr-2"></i>
                        LISTA DE CLIENTES
                    </h4>
                        <span id="clients-list-service-name" class="ml-2 text-xl text-gray-600 font-black uppercase"></span>
                </div>
                <div id="clients-list-container" class="space-y-2 max-h-[60vh] overflow-y-auto pr-2 custom-scrollbar">
                    <p class="text-sm text-gray-500 italic">Cargando...</p>
                </div>
            </div>
        </div>

        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <button type="button" class="w-full bg-[var(--color-primary)] hover:bg-gray-800 text-white font-bold py-2 px-4 rounded-lg uppercase transition-all flex items-center justify-center gap-2 shadow-sm" onclick="closePanel('viewClientsPanel')">
                <i data-lucide="x-circle" class="w-5 h-5"></i> CERRAR
            </button>
        </div>
    </div>
</div>

<div id="manageCategoriesPanel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="folder-cog" class="w-8 h-8 mr-2 text-white"></i>
                GESTIONAR CATEGORÍAS
            </h3>
            <button onclick="closePanel('manageCategoriesPanel')" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
                <i data-lucide="x" class="w-8 h-8"></i>
            </button>
        </div>

        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <div class="border-b pb-3 mb-4">
                    <h4 class="text-2xl font-black text-[var(--color-secondary)] uppercase flex items-center">
                        <i data-lucide="tag" class="w-5 h-5 mr-2"></i>
                        ADMINISTRACIÓN
                    </h4>
                </div>

                <div class="space-y-6">
                    <form id="add-category-form">
                        <label for="new-category-name" class="block text-md font-black text-black mb-1 uppercase">Nombre de la Categoría</label>
                        <div class="flex gap-2">
                            <input type="text" id="new-category-name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Ej. Contabilidad" required>
                            <button type="submit" class="bg-[var(--color-primary)] hover:bg-[var(--color-highlight)] text-white font-bold py-2 px-4 rounded-lg uppercase shadow-md transition duration-200 flex items-center justify-center">
                                <i data-lucide="plus" class="w-5 h-5"></i>
                            </button>
                        </div>
                    </form>

                    <div class="border-t pt-4">
                        <h4 class="text-sm font-bold text-gray-500 uppercase mb-3">Categorías Existentes</h4>
                        <div id="category-list-container" class="space-y-2 bg-gray-50 p-2 rounded-lg border border-gray-100 max-h-60 overflow-y-auto custom-scrollbar">
                            <p class="text-gray-500 text-sm italic">Cargando Categorías...</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <button type="button" class="w-full bg-[var(--color-primary)] hover:bg-gray-800 text-white font-bold py-2 px-4 rounded-lg uppercase transition-all flex items-center justify-center gap-2 shadow-sm" onclick="closePanel('manageCategoriesPanel')">
                <i data-lucide="x-circle" class="w-5 h-5"></i> CERRAR
            </button>
        </div>
    </div>
</div>

<div id="confirmDeleteModal" 
     class="fixed inset-0 bg-gray-900 bg-opacity-90 flex items-center justify-center hidden z-50 font-barlow"
     onclick="if(event.target === this) closeModal('confirmDeleteModal')">
    
    <div class="modal-dialog relative mx-auto p-0 border-0 shadow-2xl rounded-xl bg-white w-full max-w-sm m-4 text-center transform transition-all duration-300 scale-95 opacity-0 overflow-hidden">
        
        <div class="modal-header-container bg-[var(--color-primary)] p-6">
            <h3 class="modal-primary-title text-3xl font-black text-[var(--color-highlight)] uppercase leading-none">ADVERTENCIA</h3>
        </div>

        <div class="p-8">
            <div class="flex justify-center mb-6">
                <i data-lucide="alert-triangle" class="w-16 h-16 text-[var(--color-secondary)]"></i>
            </div>
            
            <p class="text-[var(--color-primary)] mb-6 uppercase text-lg font-bold leading-tight">
                ¿DESEAS ELIMINAR:
                <span id="confirm-item-type" class="text-[var(--color-primary)] mb-6 uppercase text-lg font-bold leading-tight"></span>
                <span id="confirm-item-name" class="font-black text-[var(--color-secondary)] text-xl inline-block uppercase"></span>?
            </p>
            
            <div class="flex flex-col sm:flex-row justify-center space-y-3 sm:space-y-0 sm:space-x-4">
                <button type="button" class="w-full bg-[var(--color-primary)] hover:bg-gray-800 text-white font-black py-2.5 px-6 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2 shadow-sm" onclick="closeModal('confirmDeleteModal')">
                    <i data-lucide="x-circle" class="w-5 h-5"></i> CANCELAR
                </button>
                
                <button type="button" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-6 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2 shadow-md" id="confirm-delete-button">
                    <i data-lucide="trash-2" class="w-5 h-5"></i> CONFIRMAR
                </button>
            </div>
            
            <p class="mt-6 uppercase text-xs font-black text-gray-500 tracking-wider">  
                ESTA ACCIÓN NO SE PUEDE DESHACER
            </p>
        </div>
    </div>
</div>

<script>
let serviciosActuales = <?php echo $servicios_json; ?>;
let allCategories = []; 
let monthlySoldServicesCount = <?php echo json_encode($monthly_sold_services_count); ?>;

// Variables globales para las instancias de Quill
let addFeaturesQuill, addDescriptionQuill;
let viewFeaturesQuill, viewDescriptionQuill;

// FUNCIÓN DE UTILIDAD BASADA EN TU editor.js para inicializar MULTIPLES editores
function initializeQuillUtility(editorId, hiddenInputId, initialContent = '') {
    const editorContainer = document.getElementById(editorId);
    const hiddenInput = document.getElementById(hiddenInputId);

    if (!editorContainer || !hiddenInput) {
        return null;
    }

    // CONFIGURACIÓN DE QUILL - Módulos copiados de tu editor.js
    const quillInstance = new Quill(editorContainer, {
        theme: 'snow', 
        placeholder: 'Escribe Aquí...',
        modules: {
            toolbar: [
                ['bold', 'italic', 'underline'],
                ['blockquote'],
                [{ 'header': [1, 2, 3, false] }],
                [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                ['clean']
            ]
        }
    });

    // Cargar contenido inicial
    if (initialContent) {
        quillInstance.clipboard.dangerouslyPasteHTML(initialContent);
    }
    
    // Sincronización con el input oculto (Tu lógica simplificada, sin contador)
    quillInstance.on('text-change', function() {
        let html = quillInstance.root.innerHTML;
        if (html === '<p><br></p>' || quillInstance.getText().trim() === '') {
            html = '';
        }
        hiddenInput.value = html;
    });
    
    // Inicializa el input oculto con el contenido.
    hiddenInput.value = initialContent || '';
    
    return quillInstance;
}

function initializeAllQuillEditors() {
    // Panel Agregar Servicio 
    addFeaturesQuill = initializeQuillUtility('add-features-quill', 'add-features-hidden');
    addDescriptionQuill = initializeQuillUtility('add-description-quill', 'add-description-hidden');
    
    // Panel Modificar Servicio
    viewFeaturesQuill = initializeQuillUtility('view-features-quill', 'view-features-hidden');
    viewDescriptionQuill = initializeQuillUtility('view-description-quill', 'view-description-hidden');
}

// Función para extraer el ID de un link de YouTube
function getYoutubeEmbedId(url) {
    if (!url) return null;
    const regExp = /^.*(youtu\.be\/|v\/|u\/\w\/|embed\/|watch\?v=|\&v=)([^#\&\?]*).*/;
    const match = url.match(regExp);
    return (match && match[2].length === 11) ? match[2] : null;
}


document.addEventListener('DOMContentLoaded', function() {
    const refs = {
        monthlySold: document.getElementById('monthly-sold-services-count'),
        totalServices: document.getElementById('total-services-count'),
        serviceSearch: document.getElementById('service-search'),
        serviceFilterCategory: document.getElementById('service-filter-category'),
        serviceGridView: document.getElementById('service-grid-view'),
        serviceListView: document.getElementById('service-list-view'),
        servicesTableBody: document.getElementById('servicesTableBody'),
        gridViewBtn: document.getElementById('grid-view-btn'),
        listViewBtn: document.getElementById('list-view-btn'),
        confirmDeleteButton: document.getElementById('confirm-delete-button'),
        confirmItemNameSpan: document.getElementById('confirm-item-name'),
        confirmItemTypeSpan: document.getElementById('confirm-item-type'),
        addServicioForm: document.getElementById('add-servicio-form'),
        viewServicioForm: document.getElementById('view-servicio-form'),
        addServiceCategorySelect: document.getElementById('servicio-categoria'),
        viewServiceCategorySelect: document.getElementById('view-servicio-categoria'),
        addCategoryForm: document.getElementById('add-category-form'),
        newCategoryNameInput: document.getElementById('new-category-name'),
        categoryListContainer: document.getElementById('category-list-container')
    };
    
    // INICIALIZAMOS EDITORES DE QUILL usando la función de utilidad
    initializeAllQuillEditors();


    let currentServiceView = 'grid';
    const panelOverlay = document.getElementById('panel-overlay');

    window.openPanel = (panelId) => {
        const panel = document.getElementById(panelId);
        if (!panel) return;
        panel.classList.remove('translate-x-full');
        panelOverlay.classList.remove('hidden');
        if (panelId === 'addServicioPanel') {
            refs.addServicioForm.reset();
            // LIMPIAR EDITORES
            addFeaturesQuill?.setText('');
            addDescriptionQuill?.setText('');
            // LIMPIAR INPUTS OCULTOS
            document.getElementById('add-features-hidden').value = '';
            document.getElementById('add-description-hidden').value = '';
            
            populateCategoryDropdowns();
        } else if (panelId === 'manageCategoriesPanel') {
            refs.newCategoryNameInput.value = '';
            renderCategoryList();
        }
        lucide.createIcons();
    };

    window.closePanel = (panelId) => {
        document.getElementById(panelId)?.classList.add('translate-x-full');
        const modalOpen = document.getElementById('confirmDeleteModal')?.classList.contains('hidden') === false;
        if (!modalOpen) {
            panelOverlay.classList.add('hidden');
        }
    };
    
    window.closeModal = (modalId) => {
        const modal = document.getElementById(modalId);
        if (!modal) return;
        const content = modal.querySelector('.modal-dialog');
        content.classList.remove('scale-100', 'opacity-100');
        content.classList.add('scale-95', 'opacity-0');
        setTimeout(() => modal.classList.add('hidden'), 300);

        const isPanelOpen = ['addServicioPanel', 'viewServicioPanel', 'manageCategoriesPanel', 'viewClientsPanel'].some(id => !document.getElementById(id)?.classList.contains('translate-x-full'));
        if (!isPanelOpen) {
            panelOverlay.classList.add('hidden');
        }
    };

    window.openModal = (modalId) => {
        const modal = document.getElementById(modalId);
        if (!modal) return;
        const content = modal.querySelector('.modal-dialog');
        modal.classList.remove('hidden');
        setTimeout(() => {
             content.classList.remove('scale-95', 'opacity-0');
             content.classList.add('scale-100', 'opacity-100');
        }, 50);
        panelOverlay.classList.remove('hidden');
        lucide.createIcons();
    };
    
    window.openConfirmDeleteModal = (itemId, itemType, itemName = 'ESTE ELEMENTO') => {
        refs.confirmDeleteButton.dataset.itemId = itemId;
        refs.confirmDeleteButton.dataset.itemType = itemType;
        refs.confirmItemNameSpan.textContent = itemName.toUpperCase(); 
        refs.confirmItemTypeSpan.textContent = itemType.toUpperCase();
        openModal('confirmDeleteModal');
    };

    window.openViewServicioModal = (servicioId) => {
        const servicio = serviciosActuales[servicioId];
        if (!servicio) { showToast('Servicio no encontrado.', 'error'); return; }
        document.getElementById('view-servicio-id').value = servicio.db_id;
        document.getElementById('view-servicio-nombre').value = servicio.nombre;
        document.getElementById('view-servicio-precio').value = servicio.precio;
        
        // CARGA DE CONTENIDO EN QUILL
        // El PHP ya no aplica htmlspecialchars, por lo que el HTML se carga limpio.
        viewFeaturesQuill?.clipboard.dangerouslyPasteHTML(servicio.features || '');
        viewDescriptionQuill?.clipboard.dangerouslyPasteHTML(servicio.descripcion || '');

        // SINCRONIZACIÓN INMEDIATA DE HIDDEN INPUTS CON CONTENIDO DE QUILL
        document.getElementById('view-features-hidden').value = servicio.features || '';
        document.getElementById('view-description-hidden').value = servicio.descripcion || '';


        // NUEVOS CAMPOS: YouTube y Paths actuales para el Update
        document.getElementById('view-servicio-youtube-link').value = servicio.youtube_link || ''; 
        document.getElementById('view-current-image-path').value = servicio.imagen || '';
        document.getElementById('view-current-gallery-images').value = servicio.gallery_images || '';

        // Previsualización de portada
        const previewDiv = document.getElementById('view-servicio-preview');
        previewDiv.innerHTML = ''; // Limpiar
        previewDiv.classList.remove('justify-center', 'items-center');

        if (servicio.youtube_link) {
            const youtubeId = getYoutubeEmbedId(servicio.youtube_link);
            if (youtubeId) {
                previewDiv.innerHTML = `<iframe class="w-full h-full rounded-lg" src="https://www.youtube.com/embed/${youtubeId}?controls=0" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>`;
            }
        } else if (servicio.imagen) {
            previewDiv.innerHTML = `<img src="uploads/services/${servicio.imagen}" alt="Portada" class="w-full h-full object-cover rounded-lg">`;
        } else {
            previewDiv.innerHTML = `<i data-lucide="image" class="w-8 h-8 text-gray-400"></i>`;
            previewDiv.classList.add('justify-center', 'items-center');
        }

        // Previsualización de Galería
        const galleryPreviewDiv = document.getElementById('view-servicio-gallery-preview');
        galleryPreviewDiv.innerHTML = '';
        if (servicio.gallery_images) {
            const images = servicio.gallery_images.split(',');
            images.forEach(img => {
                if (img) {
                    galleryPreviewDiv.innerHTML += `<img src="uploads/services/${img}" alt="Galería" class="w-16 h-16 object-cover rounded shadow border border-gray-200">`;
                }
            });
        }
        if (galleryPreviewDiv.innerHTML === '') {
            galleryPreviewDiv.innerHTML = '<p class="text-xs text-gray-400 italic">No hay fotos en la galería actual.</p>';
        }


        populateCategoryDropdowns(servicio.categoria);
        openPanel('viewServicioPanel');
        lucide.createIcons(); // Recargar íconos
    };

    async function fetchCategories() {
        try {
            const res = await fetch('db/services-categories-read.php');
            const data = await res.json();
            if (data.success) {
                allCategories = data.categories;
                populateCategoryFilter();
                populateCategoryDropdowns();
            }
        } catch (e) { console.error(e); }
    }

    function populateCategoryFilter() {
        refs.serviceFilterCategory.innerHTML = '<option value="all">TODAS LAS CATEGORÍAS</option>';
        allCategories.forEach(cat => {
            refs.serviceFilterCategory.add(new Option(cat.name.toUpperCase(), cat.name));
        });
    }

    function populateCategoryDropdowns(selected = '') {
        [refs.addServiceCategorySelect, refs.viewServiceCategorySelect].forEach(sel => {
            if(!sel) return;
            sel.innerHTML = '<option value="">Selecciona Una Categoría</option>';
            allCategories.forEach(cat => {
                const opt = new Option(cat.name, cat.name);
                if(cat.name === selected) opt.selected = true;
                sel.add(opt);
            });
        });
    }

    function copyToClipboard(text) {
        navigator.clipboard.writeText(text).then(() => {
            showToast('Link de oferta copiado', 'success');
        }, () => {
            showToast('Error al copiar', 'error');
        });
    }

    function renderServiceCard(service) {
        // LINK PUBLICO
        const publicLink = `${window.location.origin}/services-offered.php?id=${service.db_id}`;
        
        // IMAGEN / VIDEO
        const youtubeId = service.youtube_link ? getYoutubeEmbedId(service.youtube_link) : null;

        const imgHtml = youtubeId
            ? `<div class="h-40 w-full bg-black flex items-center justify-center relative"><iframe class="w-full h-full" src="https://www.youtube.com/embed/${youtubeId}?controls=0" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe></div>`
            : service.imagen
            ? `<div class="h-40 w-full bg-gray-200"><img src="uploads/services/${service.imagen}" alt="${service.nombre}" class="w-full h-full object-cover"></div>`
            : `<div class="h-20 w-full bg-[var(--color-primary)] flex items-center justify-center"><i data-lucide="image" class="w-8 h-8 text-white opacity-50"></i></div>`;

        // FEATURES PREVIEW: Renderiza el HTML de Quill directamente
        let featuresHtml = '';
        if(service.features) {
            featuresHtml = `<div class="mt-2 text-xs text-gray-500 line-clamp-2">${service.features}</div>`;
        }
        
        // DESCRIPCIÓN PREVIEW: Renderiza el HTML de Quill directamente
        let descriptionHtml = '';
        if(service.descripcion) {
             descriptionHtml = `<p class="text-gray-500 text-xs flex-grow mb-2 line-clamp-2 leading-relaxed">${service.descripcion}</p>`;
        }


        const card = document.createElement('div');
        card.className = 'bg-white rounded-xl shadow-sm hover:shadow-lg overflow-hidden flex flex-col border border-gray-200 transition-all duration-300 transform hover:-translate-y-1';
        card.dataset.servicioId = service.id;
        card.innerHTML = `
            ${imgHtml}
            <div class="bg-[var(--color-primary)] text-white p-3 flex items-center justify-between">
                <div class="flex items-center gap-2 flex-grow min-w-0">
                    <i data-lucide="clipboard-check" class="w-4 h-4 flex-shrink-0 text-[var(--color-highlight)]"></i>
                    <h3 class="font-black text-md uppercase truncate tracking-wide">${service.nombre}</h3>
                </div>
            </div>
            <div class="p-5 flex-grow flex flex-col">
                <div class="flex justify-between items-center mb-2 pb-2 border-b border-gray-100">
                     <div>
                        <p class="text-gray-400 uppercase text-[10px] font-bold">PRECIO</p>
                        <p class="text-[var(--color-secondary)] font-black text-2xl">$${service.precio.toFixed(2)}</p>
                     </div>
                     <div class="text-right">
                        <span class="inline-flex items-center bg-blue-50 text-blue-800 text-xs font-bold px-2 py-1 rounded-full border border-blue-100">
                            <i data-lucide="users" class="w-3 h-3 mr-1"></i> ${service.active_clients}
                        </span>
                     </div>
                </div>
                
                ${descriptionHtml}
                ${featuresHtml}
                
                <div class="mt-auto pt-4 flex justify-between items-center">
                    <span class="inline-block bg-gray-100 text-gray-500 text-[10px] font-bold px-2 py-1 rounded border border-gray-200 uppercase">
                        ${service.categoria || 'GENERAL'}
                    </span>
                </div>
            </div>
            <div class="bg-gray-50 p-3 flex justify-between items-center border-t border-gray-200">
                <button class="text-green-600 hover:text-green-800 p-2 rounded-lg transition-colors view-clients-btn" data-service-db-id="${service.db_id}" data-service-name="${service.nombre}" title="Ver Clientes">
                    <i data-lucide="eye" class="w-6 h-6 text-[var(--color-primary)]"></i>
                </button>
                <div class="flex space-x-2">
                     <button onclick="copyToClipboard('${publicLink}')" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2">
                        <i data-lucide="link" class="w-3 h-3 mr-1"></i> LINK
                    </button>
                    <button class="w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase sm:w-auto flex items-center justify-center gap-2 edit-servicio-btn" data-servicio-id="${service.id}">
                        <i data-lucide="link" class="w-3 h-3 mr-1"></i> EDITAR
                    </button>
                    <button class="text-green-600 hover:text-green-800 p-2 rounded-lg transition-colors shadow-sm delete-btn" data-item-type="servicio" data-item-id="${service.id}" title="ELIMINAR">
                        <i data-lucide="trash-2" class="w-6 h-6 text-[var(--color-primary)]"></i>
                    </button>
                </div>
            </div>`;
        return card;
    }

    function renderServiceTableRow(service) {
        const row = document.createElement('tr');
        row.className = 'border-b border-gray-100 hover:bg-gray-50 transition-colors';
        row.dataset.servicioId = service.id;
        row.innerHTML = `
            <td class="flex justify-between items-center md:table-cell w-full md:w-auto py-2 md:py-4 px-0 md:px-6 border-b md:border-none last:border-0" data-label="NOMBRE">
                <span class="block md:hidden font-bold text-xs text-gray-500 uppercase">NOMBRE</span>
                <span class="font-bold uppercase text-gray-800 text-right md:text-left">${service.nombre}</span>
            </td>
            <td class="flex justify-between items-center md:table-cell w-full md:w-auto py-2 md:py-4 px-0 md:px-6 border-b md:border-none last:border-0" data-label="CATEGORÍA">
                <span class="block md:hidden font-bold text-xs text-gray-500 uppercase">CATEGORÍA</span>
                <div class="text-right md:text-left text-sm text-gray-600 uppercase">${service.categoria || 'N/A'}</div>
            </td>
            <td class="flex justify-between items-center md:table-cell w-full md:w-auto py-2 md:py-4 px-0 md:px-6 border-b md:border-none last:border-0" data-label="PRECIO">
                <span class="block md:hidden font-bold text-xs text-gray-500 uppercase">PRECIO</span>
                <div class="text-right md:text-left font-bold text-[var(--color-secondary)]">$${service.precio.toFixed(2)}</div>
            </td>
            <td class="flex justify-between items-center md:table-cell w-full md:w-auto py-2 md:py-4 px-0 md:px-6 border-b md:border-none last:border-0" data-label="CLIENTES ACTIVOS">
                <span class="block md:hidden font-bold text-xs text-gray-500 uppercase">CLIENTES</span>
                <div class="text-right md:text-center">
                    <span class="inline-flex items-center bg-blue-50 text-blue-800 text-xs font-bold px-2.5 py-0.5 rounded-full border border-blue-100">
                        ${service.active_clients}
                    </span>
                </div>
            </td>
            <td class="flex justify-between items-center md:table-cell w-full md:w-auto py-2 md:py-4 px-0 md:px-6 border-b md:border-none last:border-0" data-label="ACCIONES">
                <span class="block md:hidden font-bold text-xs text-gray-500 uppercase">ACCIONES</span>
                <div class="flex justify-end md:justify-center space-x-2">
                    <button class="text-green-600 hover:text-green-800 p-1.5 hover:bg-green-50 rounded transition view-clients-btn" title="Ver Clientes" data-service-db-id="${service.db_id}" data-service-name="${service.nombre}"><i data-lucide="eye" class="w-5 h-5"></i></button>
                    <button class="text-blue-600 hover:text-blue-800 p-1.5 hover:bg-blue-50 rounded transition edit-servicio-btn" title="EDITAR" data-servicio-id="${service.id}"><i data-lucide="edit-2" class="w-5 h-5"></i></button>
                    <button class="text-white hover:text-white p-1.5 hover:bg-[var(--color-secondary)] rounded transition delete-btn" data-item-type="servicio" data-item-id="${service.id}" title="ELIMINAR"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
                </div>
            </td>`;
        return row;
    }

   function renderServices(viewType = currentServiceView) {
    const term = refs.serviceSearch.value.toLowerCase();
    const catFilter = refs.serviceFilterCategory.value;
    
    const filtered = Object.values(serviciosActuales).filter(s => {
        // En la búsqueda, verificamos el nombre
        const matchesName = s.nombre.toLowerCase().includes(term); 
        const matchesCat = catFilter === 'all' || s.categoria === catFilter;
        return matchesName && matchesCat;
    });

    refs.serviceGridView.innerHTML = '';
    refs.servicesTableBody.innerHTML = '';

    if (filtered.length === 0) {
        refs.serviceGridView.innerHTML = `<div class="col-span-full text-center py-12 bg-white rounded-xl border border-dashed border-gray-300"><p class="text-gray-400 font-medium text-lg">NO SE ENCONTRARON SERVICIOS.</p></div>`;
        refs.servicesTableBody.innerHTML = `<tr><td colspan="5" class="py-8 text-center text-gray-500 italic">NO SE ENCONTRARON SERVICIOS.</td></tr>`;
    } else {
        filtered.forEach(s => {
            if(viewType === 'grid') refs.serviceGridView.appendChild(renderServiceCard(s));
            else refs.servicesTableBody.appendChild(renderServiceTableRow(s));
        });
    }

    if (viewType === 'grid') {
        refs.serviceGridView.classList.remove('hidden');
        refs.serviceListView.classList.add('hidden');
        
        refs.gridViewBtn.classList.add('bg-white', 'text-gray-800', 'shadow-sm');
        refs.gridViewBtn.classList.remove('bg-transparent', 'text-gray-500');
        refs.listViewBtn.classList.add('bg-transparent', 'text-gray-500');
        refs.listViewBtn.classList.remove('bg-white', 'text-gray-800', 'shadow-sm');
    } else {
        refs.serviceGridView.classList.add('hidden');
        refs.serviceListView.classList.remove('hidden');
        
        refs.listViewBtn.classList.add('bg-white', 'text-gray-800', 'shadow-sm');
        refs.listViewBtn.classList.remove('bg-transparent', 'text-gray-500');
        refs.gridViewBtn.classList.add('bg-transparent', 'text-gray-500');
        refs.gridViewBtn.classList.remove('bg-white', 'text-gray-800', 'shadow-sm');
    }
    
    document.querySelectorAll('.delete-btn').forEach(b => b.addEventListener('click', (e) => {
        e.stopPropagation();
        const itemId = e.currentTarget.dataset.itemId;
        const itemType = e.currentTarget.dataset.itemType;
        let itemName = 'ESTE ELEMENTO'; 

        if (itemType === 'servicio') {
            const servicio = serviciosActuales[itemId];
            itemName = servicio ? servicio.nombre : itemName;
        } else if (itemType === 'categoria') {
            const categoria = allCategories.find(c => String(c.id) === String(itemId));
            itemName = categoria ? categoria.name : itemName;
        }
        window.openConfirmDeleteModal(itemId, itemType, itemName);
    }));
    
    document.querySelectorAll('.edit-servicio-btn').forEach(b => b.addEventListener('click', (e) => {
        e.stopPropagation(); openViewServicioModal(e.currentTarget.dataset.servicioId);
    }));
    document.querySelectorAll('.view-clients-btn').forEach(b => b.addEventListener('click', (e) => {
        e.stopPropagation(); openClientsPanel(e.currentTarget.dataset.serviceDbId, e.currentTarget.dataset.serviceName);
    }));
    
    // Asignar evento al botón de copiar
    window.copyToClipboard = copyToClipboard;

    lucide.createIcons();
}

    async function openClientsPanel(serviceId, serviceName) {
        openPanel('viewClientsPanel');
        const container = document.getElementById('clients-list-container');
        document.getElementById('clients-panel-title').textContent = `CLIENTES ASIGNADOS`;
        document.getElementById('clients-list-service-name').textContent = `(${serviceName})`;
        container.innerHTML = '<p class="text-gray-500 italic text-center p-4">Cargando clientes...</p>';

        try {
            const res = await fetch(`db/service-clients-read.php?service_id=${serviceId}`);
            const data = await res.json();
            if (data.success) {
                container.innerHTML = '';
                const count = data.clients.length;
                
                const summaryHtml = `
                    <div class="bg-blue-50 p-4 rounded-lg border border-blue-100 flex items-center justify-between mb-4">
                        <div class="flex items-center gap-3">
                             <div class="bg-white p-2 rounded-full shadow-sm text-blue-600">
                                 <i data-lucide="users" class="w-6 h-6"></i>
                             </div>
                             <span class="font-bold text-[var(--color-primary)] uppercase text-sm">Total Clientes</span>
                        </div>
                        <span class="text-3xl font-black text-[var(--color-secondary)]">${count}</span>
                    </div>`;
                
                container.insertAdjacentHTML('beforeend', summaryHtml);

                if (count > 0) {
                    const ul = document.createElement('ul');
                    ul.className = 'space-y-2';
                    data.clients.forEach(c => {
                        const li = document.createElement('li');
                        li.className = 'p-3 bg-white border border-gray-100 rounded-lg flex items-center shadow-sm hover:bg-gray-50 transition-colors';
                        li.innerHTML = `
                            <span class="w-2 w-2 rounded-full bg-green-500 mr-3"></span>
                            <span class="font-bold text-gray-700 uppercase text-sm">${c.first_name} ${c.last_name}</span>
                        `;
                        ul.appendChild(li);
                    });
                    container.appendChild(ul);
                } else {
                    container.innerHTML += '<div class="text-center py-8 bg-gray-50 rounded-lg border border-dashed border-gray-200"><p class="text-gray-400 font-medium">No hay clientes asignados a este servicio.</p></div>';
                }
                lucide.createIcons();
            } else {
                container.innerHTML = `<p class="text-white p-4 bg-[var(--color-secondary)] rounded-lg border border-[var(--color-secondary)] text-center">Error: ${data.message}</p>`;
            }
        } catch (e) { container.innerHTML = '<p class="text-white p-4 bg-[var(--color-secondary)] rounded-lg border [var(--color-secondary)] text-center">Error de conexión.</p>'; }
    }

    function renderCategoryList() {
        refs.categoryListContainer.innerHTML = '';
        if(allCategories.length === 0) {
            refs.categoryListContainer.innerHTML = '<p class="text-gray-400 text-sm italic text-center p-4">No hay categorías definidas.</p>';
            return;
        }
        allCategories.forEach(cat => {
            const div = document.createElement('div');
            div.className = 'flex justify-between items-center p-3 bg-white border border-gray-200 rounded-lg mb-2 hover:shadow-sm transition-shadow';
            div.innerHTML = `<span class="font-bold text-gray-700 uppercase text-sm">${cat.name}</span>
                <button class="text-gray-400 hover:text-white delete-category-btn p-1 hover:bg-[var(--color-secondary)] rounded transition-colors" data-item-type="categoria" data-item-id="${cat.id}" title="ELIMINAR"><i data-lucide="trash-2" class="w-4 h-4"></i></button>`;
            refs.categoryListContainer.appendChild(div);
        });
        
        refs.categoryListContainer.querySelectorAll('.delete-category-btn').forEach(b => b.addEventListener('click', (e) => {
            const itemId = e.currentTarget.dataset.itemId;
            const itemType = e.currentTarget.dataset.itemType;
            const categoria = allCategories.find(c => String(c.id) === String(itemId));
            const itemName = categoria ? categoria.name : 'ESTE ELEMENTO';
            openConfirmDeleteModal(itemId, itemType, itemName);
        }));
        lucide.createIcons();
    }

    // El form submission ahora lee los valores de los inputs ocultos sincronizados por Quill
    refs.addServicioForm?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData();
        formData.append('name', document.getElementById('servicio-nombre').value);
        formData.append('price', document.getElementById('servicio-precio').value);
        formData.append('category', document.getElementById('servicio-categoria').value);
        
        // LECTURA DE CONTENIDO DESDE INPUTS OCULTOS (Sincronizados por Quill)
        formData.append('description', document.getElementById('add-description-hidden').value);
        formData.append('features', document.getElementById('add-features-hidden').value); 
        
        formData.append('youtube_link', document.getElementById('servicio-youtube-link').value);

        // Imagen Principal
        const imageFile = document.getElementById('servicio-imagen').files[0];
        if (imageFile) {
            formData.append('image', imageFile);
        }
        
        // Galería (Limitado a 4 en el backend)
        const galleryFiles = document.getElementById('servicio-galeria-imagenes').files;
        for(let i = 0; i < galleryFiles.length; i++) { 
            formData.append('gallery_images[]', galleryFiles[i]);
        }


        try {
            const res = await fetch('db/services-create.php', { method: 'POST', body: formData });
            const data = await res.json();
            if(data.success) { 
                showToast(data.message, 'success'); 
                setTimeout(() => location.reload(), 1000); 
            } else {
                showToast(data.message, 'error');
            }
        } catch(e) { showToast('Error de conexión', 'error'); }
    });

    // El form submission ahora lee los valores de los inputs ocultos sincronizados por Quill
    refs.viewServicioForm?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const formData = new FormData();
        formData.append('id', document.getElementById('view-servicio-id').value);
        formData.append('name', document.getElementById('view-servicio-nombre').value);
        formData.append('price', document.getElementById('view-servicio-precio').value);
        formData.append('category', document.getElementById('view-servicio-categoria').value);
        
        // LECTURA DE CONTENIDO DESDE INPUTS OCULTOS (Sincronizados por Quill)
        formData.append('description', document.getElementById('view-description-hidden').value);
        formData.append('features', document.getElementById('view-features-hidden').value);
        
        formData.append('youtube_link', document.getElementById('view-servicio-youtube-link').value);

        // Paths Actuales
        formData.append('current_image_path', document.getElementById('view-current-image-path').value);
        formData.append('current_gallery_images', document.getElementById('view-current-gallery-images').value);

        // Imagen Principal (Nueva)
        const imageFile = document.getElementById('view-servicio-imagen').files[0];
        if (imageFile) {
            formData.append('image', imageFile);
        }
        
        // Galería (Nueva)
        const galleryFiles = document.getElementById('view-servicio-galeria-imagenes').files;
        for(let i = 0; i < galleryFiles.length; i++) {
            formData.append('gallery_images[]', galleryFiles[i]);
        }

        try {
            const res = await fetch('db/services-update.php', { method: 'POST', body: formData });
            const data = await res.json();
            if(data.success) { 
                showToast(data.message, 'success'); 
                setTimeout(() => location.reload(), 1000); 
            } else {
                showToast(data.message, 'error');
            }
        } catch(e) { showToast('Error de conexión', 'error'); }
    });

    refs.addCategoryForm?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const name = refs.newCategoryNameInput.value.trim();
        if(!name) return showToast('Nombre requerido', 'warning');
        try {
            const res = await fetch('db/services-categories-create.php', { method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({name}) });
            const data = await res.json();
            if(data.success) {
                showToast('Categoría creada', 'success');
                refs.newCategoryNameInput.value = '';
                await fetchCategories();
                renderCategoryList();
            } else showToast(data.message, 'error');
        } catch(e) { showToast('Error de conexión', 'error'); }
    });

    refs.confirmDeleteButton?.addEventListener('click', async function() {
        const id = this.dataset.itemId;
        const type = this.dataset.itemType;
        const url = type === 'servicio' ? 'db/services-delete.php' : 'db/services-categories-delete.php';
        closeModal('confirmDeleteModal');
        
        try {
            const res = await fetch(url, { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify({id}) });
            const data = await res.json();
            if(data.success) { 
                showToast(data.message, 'success'); 
                setTimeout(() => location.reload(), 1000); 
            } else {
                showToast(data.message, 'error');
            }
        } catch(e) { showToast('Error al eliminar', 'error'); }
    });

    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebarOverlay = document.getElementById('sidebar-overlay');
    const sidebar = document.getElementById('sidebar'); 
    
    if(mobileMenuButton) mobileMenuButton.addEventListener('click', () => { sidebar?.classList.toggle('-translate-x-full'); sidebarOverlay?.classList.toggle('hidden'); });
    if(sidebarOverlay) sidebarOverlay.addEventListener('click', () => { sidebar?.classList.add('-translate-x-full'); sidebarOverlay.classList.add('hidden'); });
    panelOverlay.addEventListener('click', () => { 
        const modalOpen = document.getElementById('confirmDeleteModal')?.classList.contains('hidden') === false;
        if (!modalOpen) {
            closePanel('addServicioPanel'); 
            closePanel('viewServicioPanel'); 
            closePanel('manageCategoriesPanel'); 
            closePanel('viewClientsPanel'); 
        }
    });
    
    refs.serviceSearch?.addEventListener('input', () => renderServices(currentServiceView));
    refs.serviceFilterCategory?.addEventListener('change', () => renderServices(currentServiceView));
    refs.gridViewBtn?.addEventListener('click', () => { currentServiceView = 'grid'; renderServices('grid'); });
    refs.listViewBtn?.addEventListener('click', () => { currentServiceView = 'list'; renderServices('list'); });

    fetchCategories();
    if(refs.totalServices) refs.totalServices.textContent = Object.keys(serviciosActuales).length;
    if(refs.monthlySold) refs.monthlySold.textContent = monthlySoldServicesCount;
    renderServices(currentServiceView);
});
</script>
<script src="files/toast.js"></script>
</body>
</html>