<?php
declare(strict_types=1);

if (session_status() === PHP_SESSION_NONE) {
    ob_start();
}

// LÓGICA DE NEGOCIO Y BACKEND (PHP)
require_once(__DIR__ . '/core/init.php');
require_once 'files/motivational-quotes.php';
require_once 'media/videos-crm.php';

$csrf_token = $_SESSION['csrf_token'] ?? '';

// CONSTANTES DE BACKEND (URLs, rutas)
const API_TASK_CREATE_URL = 'db/tasks-create.php';
const API_TASK_UPDATE_URL = 'db/tasks-update.php';
const API_TASK_DELETE_URL = 'db/tasks-delete.php';
const API_CONTACT_SEARCH_URL = 'db/contact-search.php';
const API_DASHBOARD_READ_URL = 'db/dashboard-read.php';

function getDailyQuote(array $quotes): array
{
    $defaultQuote = ['quote' => 'La persistencia es la clave del éxito.', 'author' => 'OrozCO'];
    
    if (empty($quotes)) {
        return $defaultQuote;
    }
    
    $dayOfYear = (int)date('z');
    $quoteIndex = $dayOfYear % count($quotes);
    
    return $quotes[$quoteIndex];
}

$motivational_quotes = $motivational_quotes ?? [];
$dailyQuoteData = getDailyQuote($motivational_quotes);
$daily_quote = $dailyQuoteData['quote'];
$daily_author = $dailyQuoteData['author'];

$video_data = get_video_by_id('dashboard');

if (ob_get_level() > 0) { 
    ob_end_clean(); 
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Tu Centro De Control Empresarial <?php echo htmlspecialchars($branding['full_title'] ?? 'CRM'); ?></title>
    <meta name="robots" content="noindex, nofollow">
    
    <link rel="icon" type="image/png" href="<?php echo htmlspecialchars($branding['favicon'] ?? ''); ?>">
    <link rel="apple-touch-icon" href="<?php echo htmlspecialchars($branding['favicon'] ?? ''); ?>">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="<?php echo htmlspecialchars($google_font_url ?? ''); ?>">

    <?php include 'files/gtm-head.php'; ?>
    
    <script src="https://unpkg.com/lucide@latest"></script>
    <link rel="stylesheet" href="style.css"> 
    <script src="files/header-manager.js"></script>
</head>

<body data-page-title="<?php echo htmlspecialchars($branding['crm_name'] ?? ''); ?> Tu Centro De Control Empresarial"
      data-page-subtitle="La Plataforma Todo En Uno Para Impulsar Tu Negocio"
      data-page-icon="layout-dashboard">
    
    <div id="toast-container" class="fixed top-4 right-4 z-[999] space-y-3"></div>
    
<?php include 'files/gtm-body.php'; ?>  

<div class="relative min-h-screen md:flex">
    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>
    <div id="task-panel-overlay" class="fixed inset-0 bg-black/60 z-40 hidden transition-opacity duration-300"></div> 
    
    <?php include 'menu.php'; ?>
    
    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
            
            <div id="videotutorial-container" class="hidden flex justify-end p-0">
                <button id="show-video-btn" class="inline-flex items-center gap-3 text-base font-bold text-white bg-[var(--color-primary)] rounded-lg shadow-lg shadow-[var(--color-primary)]/20 py-3 px-6 transform hover:-translate-y-1 hover:shadow-xl transition-all duration-300 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[var(--color-secondary)]">
                    <i data-lucide="eye" class="w-5 h-5"></i>
                    <span>VER VIDEO PRESENTACIÓN</span>
                </button>
            </div>
        </header>
        <div id="content-area" class="p-4 md:p-8 space-y-8">
            
            <section id="videotutorial-section" class="p-6 md:p-10 rounded-xl shadow-2xl overflow-hidden hidden bg-[var(--color-primary)] relative">
                <div class="blob bl1 absolute inset-0 z-0"></div>
                <div class="blob bl2 absolute inset-0 z-0"></div>
                
                <div class="relative z-10 grid grid-cols-1 lg:grid-cols-2 gap-8 items-center">
                    <div>
                        <span class="inline-flex items-center gap-2 chip bg-black/30 text-white font-bold px-3 py-1 rounded-full text-sm">
                            <i data-lucide="laptop-minimal" class="w-4 h-4 text-[var(--color-highlight)]"></i> Bienvenido A Tu Sistema <?php echo htmlspecialchars($branding['crm_name'] ?? ''); ?>
                        </span>
                        <h1 class="mt-3 text-3xl md:text-4xl font-extrabold leading-tight text-[var(--color-highlight)]">El CRM Para El Manejo Y Crecimiento De Tu Negocio</h1>
                        <p class="mt-2 text-white/90 max-w-2xl text-lg">Centraliza tus clientes, citas y tareas. Automatiza recordatorios y mantén tu operación avanzando todos los días.</p>
                        <div class="mt-4 flex flex-wrap gap-2">
                            <a href="#inicio" class="inline-flex items-center gap-2 text-[var(--color-primary)] bg-[var(--color-highlight)] rounded-lg shadow-md hover:bg-[var(--color-highlight)]/90 font-black py-2 px-4 transition-all duration-300 text-sm hero-cta">
                                <i data-lucide="list-todo" class="w-5 h-5"></i> TAREAS PENDIENTES
                            </a>
                            <a href="/agenda" class="inline-flex items-center gap-2 text-white bg-[var(--color-primary)] rounded-lg shadow-md hover:bg-[var(--color-primary)]/90 font-black py-2 px-4 transition-all duration-300 text-sm hero-cta secondary">
                                <i data-lucide="calendar" class="w-6 h-6"></i> ABRIR TU AGENDA
                            </a>
                        </div>
                    </div>
                    <div>
                        
                        <div class="w-full">
                            <div class="bg-black/40 rounded-xl ring-1 ring-white/10 overflow-hidden">
                                <div id="video-player-wrapper" class="relative w-full aspect-video">
                                    <video id="videoPlayer" class="absolute top-0 left-0 w-full h-full object-cover" controls controlslist="nodownload" preload="none" poster="<?= htmlspecialchars($video_data['poster_url'] ?? '') ?>">
                                        <source src="<?= htmlspecialchars($video_data['video_url'] ?? '') ?>" type="video/mp4">
                                        TU NAVEGADOR NO SOPORTA LA ETIQUETA DE VIDEO
                                    </video>
                                    <div id="playOverlay" class="absolute inset-0 flex justify-center items-center cursor-pointer group">
                                        <img src="<?= htmlspecialchars($video_data['poster_url'] ?? '') ?>" alt="VideoLeccion - OrozCO" class="absolute inset-0 w-full h-full object-cover">
                                        <div class="absolute inset-0 w-full h-full bg-black/70 transition-colors group-hover:bg-black/30"></div>
                                        <i data-lucide="play" class="relative w-16 h-16 text-white opacity-90 group-hover:opacity-100 transition-opacity"></i>
                                    </div>
                                </div>
                                <div class="p-4 text-white/90 text-sm flex flex-col md:flex-row md:items-center justify-between gap-2">
                                    <span class="inline-flex items-center gap-2">
                                        <i data-lucide="play-circle" class="w-4 h-4 text-[var(--color-highlight)]"></i>
                                        <b>VIDEO:</b> <?= htmlspecialchars($video_data['title'] ?? 'Presentación') ?>
                                    </span>
                                    <button id="hide-video-btn" class="inline-flex items-center gap-2 text-xs font-black text-[var(--color-highlight)] hover:text-[var(--color-highlight)] transition-colors py-2 md:py-1 px-2 rounded-md justify-center md:justify-end w-full md:w-auto bg-white/10 md:bg-transparent min-h-[44px] md:min-h-0"
                                    title="Ocultar Video">
                                    OCULTAR VIDEO
                                    <i data-lucide="square-x" class="w-4 h-4"></i>
                                </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            <section class="mb-6">
                <div class="relative p-6 rounded-xl shadow-lg overflow-hidden bg-gradient-to-r from-[var(--color-primary)] to-[var(--color-primary)] text-white border-l-4 border-[var(--color-highlight)]">
                    
                    <div class="relative z-10 flex items-center gap-4 sm:gap-6">
                        
                        <div class="flex-shrink-0 flex items-center justify-center w-16 h-16 sm:w-20 sm:h-20 rounded-full bg-white/10 ring-2 ring-white/20">
                            <i data-lucide="megaphone" class="w-8 h-8 sm:w-10 sm:h-10 text-[var(--color-highlight)]"></i>
                        </div>
                        
                        <div class="flex-grow text-left">
                            <p class="text-lg md:text-lg italic font-bold leading-tight">
                            "<?php echo htmlspecialchars($daily_quote, ENT_QUOTES, 'UTF-8'); ?>"
                            </p>
                            
                            <p class="text-base font-semibold text-[var(--color-highlight)] mt-4 uppercase">
                                — <?php echo htmlspecialchars($daily_author, ENT_QUOTES, 'UTF-8'); ?>
                            </p>
                        </div>
                    </div>
                    
                </div>
            </section>

            <section id="inicio" class="dashboard-section">
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-2 lg:gap-6">
                    
                    <div class="stat-card bg-white p-3 lg:p-6 rounded-xl shadow-md flex items-center border-l-4 border-[var(--color-highlight)]">
                        <i data-lucide="users" class="w-8 h-8 lg:w-12 lg:h-12 text-[var(--color-secondary)] shrink-0"></i>
                        <div class="flex-1 flex justify-between items-center ml-3 lg:ml-4">
                            <h3 class="text-md lg:text-lg font-black text-gray-500 leading-tight uppercase">LEADS EN SEGUIMIENTO</h3>
                            <p id="stat-tracking-prospects" class="text-5xl lg:text-5xl font-black text-[var(--color-primary)]">0</p>
                        </div>
                    </div>

                    <div class="stat-card bg-white p-3 lg:p-6 rounded-xl shadow-md flex items-center border-l-4 border-[var(--color-highlight)]">
                        <i data-lucide="clipboard-check" class="w-10 h-10 lg:w-14 lg:h-14 text-[var(--color-secondary)] shrink-0"></i>
                        <div class="flex-1 flex justify-between items-center ml-3 lg:ml-4">
                            <h3 class="text-md lg:text-lg font-black text-gray-500 leading-tight uppercase">
                                TAREAS PENDIENTES
                            </h3>
                            <p id="stat-pending-tasks" class="text-5xl lg:text-5xl font-black text-[var(--color-primary)]">
                                0
                            </p>
                        </div>
                    </div>

                    <div class="stat-card bg-white p-3 lg:p-6 rounded-xl shadow-md flex items-center border-l-4 border-[var(--color-highlight)]">
                        <i data-lucide="file-text" class="w-10 h-10 lg:w-14 lg:h-14 text-[var(--color-secondary)] shrink-0"></i>
                        <div class="flex-1 flex justify-between items-center ml-3 lg:ml-4">
                            <h3 class="text-md lg:text-lg font-black text-gray-500 leading-tight uppercase">
                                ESTIMADOS ENVIADOS
                            </h3>
                            <p id="stat-pending-estimates" class="text-5xl lg:text-5xl font-black text-[var(--color-primary)]">
                                0
                            </p>
                        </div>
                    </div>
                    
                </div>
            </section>

            <section id="tasks-section" class="dashboard-section mt-8">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <div class="bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="clipboard-check" class="w-7 h-7 text-[var(--color-secondary)]"></i> TAREAS PENDIENTES
                        </h3>
                        <ul id="pending-tasks-list" class="space-y-3 mb-6"></ul>
                        <button id="add-task-button" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                            <i data-lucide="plus-circle" class="w-5 h-5"></i> AÑADIR NUEVA TAREA
                        </button>
                    </div>
                    <div class="bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="check-circle" class="w-7 h-7 text-green-500"></i> TAREAS COMPLETADAS
                        </h3>
                        <ul id="completed-tasks-list" class="space-y-3 mb-6"></ul>
                    </div>
                </div>
            </section>

            <section id="main-content" class="dashboard-section mt-8">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <div class="lg:col-span-1 bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="calendar-check" class="w-7 h-7 text-[var(--color-secondary)]"></i> CITAS DE HOY / PRÓXIMAS
                        </h3>
                        <ul id="upcoming-appointments-list" class="space-y-2"></ul> 
                        <button id="add-appointment-button" class="mt-6 w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                            <i data-lucide="plus-circle" class="w-5 h-5"></i> AGENDAR NUEVA CITA
                        </button>
                    </div>
                    <div class="lg:col-span-1 bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="bell" class="w-7 h-7 text-[var(--color-secondary)]"></i> PROSPECTOS EN SEGUIMIENTO
                        </h3>
                        <div id="lista-prospectos-seguimiento" class="space-y-2"></div> 
                        <button id="add-prospect-button" class="mt-6 w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                            <i data-lucide="plus-circle" class="w-5 h-5"></i> AÑADIR NUEVO PROSPECTO
                        </button>
                    </div>
                </div>
            </section>
            
            <section id="finances-tracking" class="dashboard-section mt-8">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <div class="lg:col-span-1 bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="file-text" class="w-7 h-7 text-[var(--color-secondary)]"></i> ESTIMADOS PARA SEGUIMIENTO
                        </h3>
                        <ul id="pending-estimates-list" class="space-y-4"></ul>
                        <button id="view-all-estimates-button" class="mt-6 w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                            <i data-lucide="eye" class="w-5 h-5"></i> VER TODOS LOS ESTIMADOS
                        </button>
                    </div>
                    <div class="lg:col-span-1 bg-white p-6 rounded-xl shadow-md">
                        <h3 class="text-2xl font-black text-[var(--color-primary)] mb-4 flex items-center gap-2 uppercase">
                            <i data-lucide="wallet" class="w-7 h-7 text-[var(--color-secondary)]"></i> FACTURAS VENCIDAS POR COBRAR
                        </h3>
                        <div id="overdue-invoices-list" class="space-y-4"></div>
                        <button id="view-all-invoices-button" class="mt-6 w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                            <i data-lucide="file-check-2" class="w-5 h-5"></i> VER TODAS LAS FACTURAS
                        </button>
                        <input type="hidden" id="csrf-token-php" value="<?= htmlspecialchars($csrf_token) ?>">
                        <input type="hidden" id="api-task-create-url" value="<?= htmlspecialchars(API_TASK_CREATE_URL) ?>">
                        <input type="hidden" id="api-task-update-url" value="<?= htmlspecialchars(API_TASK_UPDATE_URL) ?>">
                        <input type="hidden" id="api-task-delete-url" value="<?= htmlspecialchars(API_TASK_DELETE_URL) ?>">
                        <input type="hidden" id="api-contact-search-url" value="<?= htmlspecialchars(API_CONTACT_SEARCH_URL) ?>">
                        <input type="hidden" id="api-dashboard-read-url" value="<?= htmlspecialchars(API_DASHBOARD_READ_URL) ?>">
                    </div>
                </div>
            </section>

            <?php include 'push.php'; ?>
        </div>
    </main>
    
</div>

<div id="task-panel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="notebook-text" class="w-8 h-8 mr-2 text-white"></i> 
                AGREGAR TAREA
            </h3>
            <button id="close-task-panel" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">  
                <i data-lucide="x" class="w-8 h-8"></i>  
            </button>  
        </div>
        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">          
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <form id="add-task-form" class="space-y-4">
                    <div>
                        <label for="add-task-title" class="block text-md font-bold text-gray-600 mb-1 uppercase">Título De La Tarea</label>
                        <input type="text" id="add-task-title" name="title" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Eje. Enviar Cotización" required>
                    </div>
                    <div class="relative">
                        <label for="add-task-client-name" class="block text-md font-bold text-gray-600 mb-1 uppercase">Cliente (Opcional)</label>
                        <input type="text" id="add-task-client-name" name="client_name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Buscar Cliente">
                        <input type="hidden" id="add-task-client-id" name="client_id">
                        <div id="add-client-suggestions" class="absolute z-50 w-full bg-white border border-gray-200 shadow-lg max-h-[150px] overflow-y-auto mt-1 hidden"></div>
                    </div>
                    <div class="relative">
                        <label for="add-task-prospect-name" class="block text-md font-bold text-gray-600 mb-1 uppercase">Prospecto (Opcional)</label>
                        <input type="text" id="add-task-prospect-name" name="prospect_name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Buscar Prospecto">
                        <input type="hidden" id="add-task-prospect-id" name="prospect_id">
                        <div id="add-prospect-suggestions" class="absolute z-50 w-full bg-white border border-gray-200 shadow-lg max-h-[150px] overflow-y-auto mt-1 hidden"></div>
                    </div>
                    <div>
                        <label for="add-task-description" class="block text-md font-bold text-gray-600 mb-1 uppercase">Notas Adicionales</label>
                        <textarea id="add-task-description" name="description" rows="3" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm"></textarea>
                    </div>
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label for="add-task-due-date" class="block text-md font-bold text-gray-600 mb-1 uppercase">Fecha Vencimiento</label>
                            <input type="date" id="add-task-due-date" name="dueDate" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" required>
                        </div>
                        <div>
                            <label for="add-task-priority" class="block text-md font-bold text-gray-600 mb-1 uppercase">Prioridad</label>
                            <select id="add-task-priority" name="priority" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm bg-white">
                                <option value="Alta">Alta</option>
                                <option value="Media" selected>Media</option>
                                <option value="Baja">Baja</option>
                            </select>
                        </div>
                    </div>
                </form>
            </div>
        </div>
        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <div class="grid grid-cols-2 gap-3">
                <button type="button" id="panel-cancel-btn" class="w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                    <i data-lucide="save-off" class="w-5 h-5"></i> CANCELAR
                </button>
                <button type="submit" form="add-task-form" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">  
                    <i data-lucide="save" class="w-5 h-5"></i> GUARDAR
                </button>  
            </div>
            <input type="hidden" id="add-task-csrf-token" name="csrf_token" form="add-task-form" value="<?= htmlspecialchars($csrf_token) ?>">
        </div>
    </div>
</div>

<div id="edit-task-panel" class="fixed top-0 right-0 h-full w-full lg:w-1/3 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    <div class="flex flex-col h-full">
        <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
            <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
                <i data-lucide="notebook-pen" class="w-8 h-8 mr-2 text-white"></i>
                EDITAR TAREA
            </h3>
            <button id="close-edit-task-panel" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
                <i data-lucide="x" class="w-8 h-8"></i>
            </button>
        </div>
        <div class="flex-grow overflow-y-auto p-4 space-y-6 relative">
            <div class="bg-white p-4 rounded-xl shadow-md transition duration-300">
                <form id="edit-task-form" class="space-y-4">
                    <input type="hidden" id="edit-task-id" name="id">
                    <div>
                        <label for="edit-task-title" class="block text-md font-bold text-gray-600 mb-1 uppercase">Título De La Tarea</label>
                        <input type="text" id="edit-task-title" name="title" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Eje. Llamar A Cliente" required>
                    </div>
                    <div class="relative">
                        <label for="edit-task-client-name" class="block text-md font-bold text-gray-600 mb-1 uppercase">Cliente (Opcional)</label>
                        <input type="text" id="edit-task-client-name" name="client_name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Buscar Cliente">
                        <input type="hidden" id="edit-task-client-id" name="client_id">
                        <div id="edit-client-suggestions" class="absolute z-50 w-full bg-white border border-gray-200 shadow-lg max-h-[150px] overflow-y-auto mt-1 hidden"></div>
                    </div>
                    <div class="relative">
                        <label for="edit-task-prospect-name" class="block text-md font-bold text-gray-600 mb-1 uppercase">Prospecto (Opcional)</label>
                        <input type="text" id="edit-task-prospect-name" name="prospect_name" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" placeholder="Buscar Prospecto">
                        <input type="hidden" id="edit-task-prospect-id" name="prospect_id">
                        <div id="edit-prospect-suggestions" class="absolute z-50 w-full bg-white border border-gray-200 shadow-lg max-h-[150px] overflow-y-auto mt-1 hidden"></div>
                    </div>
                    <div>
                        <label for="edit-task-description" class="block text-md font-bold text-gray-600 mb-1 uppercase">Notas Adicionales</label>
                        <textarea id="edit-task-description" name="description" rows="3" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm"></textarea>
                    </div>
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label for="edit-task-due-date" class="block text-md font-bold text-gray-600 mb-1 uppercase">Fecha Vencimiento</label>
                            <input type="date" id="edit-task-due-date" name="dueDate" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm" required>
                        </div>
                        <div>
                            <label for="edit-task-priority" class="block text-md font-bold text-gray-600 mb-1 uppercase">Prioridad</label>
                            <select id="edit-task-priority" name="priority" class="w-full p-2 border border-gray-300 rounded-lg focus:border-[var(--color-highlight)] focus:ring-[var(--color-highlight)] focus:ring-1 transition duration-150 text-sm bg-white">
                                <option value="Alta">Alta</option>
                                <option value="Media">Media</option>
                                <option value="Baja">Baja</option>
                            </select>
                        </div>
                    </div>
                </form>
            </div>
        </div>
        <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10">  
            <div class="grid grid-cols-2 gap-3">
                <button type="button" id="edit-panel-cancel-btn" class="w-full bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                    <i data-lucide="save-off" class="w-5 h-5"></i> CANCELAR
                </button>
                <button type="submit" form="edit-task-form" class="w-full bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase flex items-center justify-center gap-2 transition duration-150">
                    <i data-lucide="save" class="w-5 h-5"></i> ACTUALIZAR TAREA
                </button>
            </div>
            <input type="hidden" id="edit-task-csrf-token" name="csrf_token" form="edit-task-form" value="<?= htmlspecialchars($csrf_token) ?>">
        </div>
    </div>
</div>

<div id="confirmDeleteModal" 
     class="fixed inset-0 bg-gray-900 bg-opacity-90 flex items-center justify-center hidden z-50">
    <div id="confirm-modal-box" class="bg-white rounded-xl shadow-2xl w-full max-w-sm m-4 transform transition-all duration-300 scale-95 opacity-0 text-center">
        <div class="rounded-t-xl pt-6 bg-[var(--color-primary)] pb-4">
            <h3 class="text-4xl font-black text-[var(--color-highlight)] uppercase leading-none">ADVERTENCIA</h3>
        </div>

        <div class="p-8">
            <div class="flex justify-center mb-6">
                <i data-lucide="alert-triangle" class="w-16 h-16 text-[var(--color-secondary)]"></i>
            </div>

            <p class="text-[var(--color-primary)] mb-6 uppercase text-lg font-bold leading-tight">
                ¿ESTÁS SEGURO DE ELIMINAR:
                <br>
                <span id="delete-item-name" class="font-black text-[var(--color-secondary)] text-xl block mt-2 uppercase"></span>
            </p>

            <div class="flex flex-col sm:flex-row justify-center space-y-2 sm:space-y-0 sm:space-x-4">
                <button type="button" class="bg-[var(--color-primary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase w-full flex items-center justify-center gap-2 transition duration-150" id="cancel-delete-button">
                    <i data-lucide="x-circle" class="w-5 h-5"></i> CANCELAR
                </button>
                
                <button type="button" class="bg-[var(--color-secondary)] hover:opacity-90 text-white font-black py-2.5 px-4 rounded-lg uppercase w-full flex items-center justify-center gap-2 transition duration-150" id="confirm-delete-button" data-item-id="" data-item-type="">
                    <i data-lucide="trash-2" class="w-5 h-5"></i> ELIMINAR
                </button>
            </div>

            <p class="mt-6 uppercase text-xs font-black text-gray-500 tracking-wider">  
                ESTA ACCIÓN NO SE PUEDE DESHACER
            </p>
        </div>
    </div>
</div>

<div id="toolsModal" class="fixed inset-0 bg-gray-900 bg-opacity-90 flex items-center justify-center hidden z-[100]">
    <div id="tools-modal-box" class="bg-white rounded-xl shadow-2xl w-full max-w-2xl m-4 transform transition-all duration-300 scale-95 opacity-0 text-center">
        <div class="rounded-t-xl pt-6 bg-[var(--color-primary)] pb-4 flex justify-between items-start px-6">
            <h3 class="text-4xl font-black text-[var(--color-highlight)] uppercase leading-none flex items-center">
                <i data-lucide="brain" class="w-8 h-8 mr-2 inline-block text-white"></i>
                HERRAMIENTAS DE ANÁLISIS
            </h3>
            <button type="button" id="closeToolsModalBtn" class="bg-[var(--color-secondary)] text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200" title="Cerrar">
                <i data-lucide="x-circle" class="w-8 h-8"></i>
            </button>
        </div>

        <div class="p-8">
            <p class="text-[var(--color-primary)] mb-6 text-lg uppercase font-black">Selecciona una de las herramientas</p>

            <div class="grid grid-cols-2 gap-4 sm:grid-cols-3">

                <a href="full-clients-profile.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="users" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Perfil Clientes</p>
                    </div>
                </a>

                <a href="full-financial-report.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="wallet" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Financiero</p>
                    </div>
                </a>

                <a href="full-growth-dashboard.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="trending-up" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Crecimiento</p>
                    </div>
                </a>

                <a href="full-leads-profile.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="bell" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Prospectos</p>
                    </div>
                </a>

                <a href="full-routes-profile.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="map-pin" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Rutas</p>
                    </div>
                </a>

                <a href="full-staff-profile.php" class="tool-card group rounded-xl shadow-2xl transition duration-300 transform hover:scale-105">
                    <div class="p-4 rounded-lg bg-[var(--color-primary)]">
                        <i data-lucide="user-check" class="w-12 h-12 mx-auto text-[var(--color-highlight)]"></i>
                        <p class="text-md font-black text-white uppercase mt-2">Empleados</p>
                    </div>
                </a>
            </div>
        </div>
    </div>
</div>

<button id="floating-tools-btn" class="fixed bottom-6 right-6 w-14 h-14 bg-[var(--color-secondary)] text-white rounded-full shadow-xl z-40 flex items-center justify-center animate-pulse focus:ring-4 focus:ring-[var(--color-highlight)] hover:scale-110 hover:bg-[var(--color-primary)] ring-4 ring-[var(--color-highlight)] ring-opacity-70 hover:animate-none focus:outline-none transition duration-300" title="Acceso Rápido a Herramientas">
   <i data-lucide="rocket" class="w-7 h-7"></i>
</button>

<template id="task-template">
    <li class="task-list-item p-2 rounded-lg transition-colors duration-200 shadow-sm border border-gray-200">
        <div class="flex justify-between items-start">
            <div class="flex-1 min-w-0">
                <p class="task-title-text text-sm font-black flex items-center truncate">
                    <span data-task-field="title"></span>
                    <span class="task-status-badge ml-1 px-1 py-0 text-xs font-bold rounded"></span>
                </p>
                <p class="task-contact-info text-md font-black uppercase text-gray-700 flex items-center mt-0.5 hidden">
                    <i data-lucide="shield-user" class="w-6 h-6 mr-1"></i> 
                    <span data-task-field="contact-name"></span>
                </p>
                <p class="task-description-text text-md text-gray-500 line-clamp-3 hidden">
                    <span data-task-field="description"></span>
                </p>
            </div>
            <div class="flex-shrink-0 flex flex-col items-end space-y-1 ml-2">
                <div class="flex items-center space-x-1 task-action-buttons" data-task-actions>
                    </div>
                <div class="flex items-center space-x-2 text-xs font-semibold">
                    <span class="task-due-date text-gray-600 flex items-center uppercase font-black hidden" data-task-field="due-date-container">
                         <i data-lucide="calendar" class="w-4 h-4 mr-0.5"></i> 
                         <span data-task-field="due-date"></span>
                    </span>
                    <span class="task-priority-badge px-1 py-0 text-xs font-bold text-white uppercase rounded">
                        <span data-task-field="priority"></span>
                    </span>
                </div>
            </div>
        </div>
    </li>
</template>

<template id="task-buttons-pending">
    <button class="text-gray-400 hover:text-[var(--color-primary)] edit-task-btn transition duration-200" data-action="edit" title="Editar Tarea"><i data-lucide="edit" class="w-6 h-6"></i></button>
    <button class="text-gray-400 hover:text-green-600 complete-task-btn transition duration-200" data-action="complete" title="Marcar como Completada"><i data-lucide="check-circle" class="w-6 h-6"></i></button>
</template>

<template id="task-buttons-completed">
    <button class="text-gray-400 hover:text-[var(--color-primary)] uncomplete-task-btn transition duration-200" data-action="uncomplete" title="Marcar como Pendiente"><i data-lucide="undo-2" class="w-6 h-6"></i></button>
</template>

<template id="task-button-delete">
    <button class="text-gray-400 hover:text-[var(--color-secondary)] delete-task-btn transition duration-200" data-action="delete" title="Eliminar Tarea"><i data-lucide="trash-2" class="w-6 h-6"></i></button>
</template>


<script src="files/toast.js"></script>
<script>
/**
 * Bloque JavaScript (Refactorizado y Unificado)
 * Rol: Lógica de Comportamiento (SoC) y Mapeo de Estado (STYLE_MAP).
 */
(function() {
    'use strict';
    
    // ** ESTADO GLOBAL **
    let DOMElements = {}; 
    let APP_DATA = {
        summaryStats: { pendingAppointments: 0, trackingProspects: 0, pendingTasks: 0, pendingEstimates: 0, overdueInvoices: 0 },
        upcomingAppointments: [],
        trackingProspectsList: [],
        pendingEstimatesList: [],  
        overdueInvoicesList: [],
        allTasks: []
    };
    let searchTimeout;

    // 1. CONFIGURACIÓN CENTRALIZADA (Extrae APIs y Tokens del DOM)
    const CONFIG = {};
    const loadApiConstants = () => {
        CONFIG.CSRF_TOKEN = document.getElementById('csrf-token-php')?.value || '';
        CONFIG.API_TASK_CREATE_URL = document.getElementById('api-task-create-url')?.value || 'db/tasks-create.php';
        CONFIG.API_TASK_UPDATE_URL = document.getElementById('api-task-update-url')?.value || 'db/tasks-update.php';
        CONFIG.API_TASK_DELETE_URL = document.getElementById('api-task-delete-url')?.value || 'db/tasks-delete.php';
        CONFIG.API_CONTACT_SEARCH_URL = document.getElementById('api-contact-search-url')?.value || 'db/contact-search.php'; 
        CONFIG.API_DASHBOARD_READ_URL = document.getElementById('api-dashboard-read-url')?.value || 'db/dashboard-read.php';
    };

    // Mapeo de Estado a Clase CSS (Centralizado para fácil cambio de diseño)
    const STYLE_MAP = {
        TASK: {
            DEFAULT: { 
                TITLE: 'text-gray-800', BG: 'bg-white', 
                BORDER: 'border-l-4 border-[var(--color-primary)]', BADGE: '' 
            },
            COMPLETED: { 
                TITLE: 'text-green-600 line-through', BG: 'bg-green-50', 
                BORDER: 'border-l-4 border-green-500', 
                BADGE: '<span class="bg-green-200 text-green-800">COMPLETADA</span>' 
            },
            OVERDUE: (dateText) => ({
                TITLE: 'text-[var(--color-secondary)] font-semibold', BG: 'bg-red-50', 
                BORDER: 'border-l-4 border-[var(--color-secondary)]', 
                BADGE: `<span class="bg-[var(--color-secondary)] text-white font-black">ATRASADA</span><span class="bg-[var(--color-primary)] text-white">(${dateText})</span>`
            }),
            TODAY: (dateText) => ({
                TITLE: 'text-[var(--color-primary)] font-semibold', BG: 'bg-yellow-50', 
                BORDER: 'border-l-4 border-yellow-500',  
                BADGE: `<span class="bg-yellow-200 text-yellow-800">HOY (${dateText})</span>` 
            }),
            PRIORITY: {
                Alta: 'bg-[var(--color-secondary)]', 
                Media: 'bg-orange-500', 
                Baja: 'bg-[var(--color-primary)]'
            }
        },
        LEAD: {
            'Proposal Sent': { ICON: 'text-blue-500', BG: 'bg-blue-50', BORDER: 'border-l-4 border-blue-500' }, 
            'Qualified': { ICON: 'text-blue-500', BG: 'bg-blue-50', BORDER: 'border-l-4 border-blue-500' }, 
            'Negotiation': { ICON: 'text-green-500', BG: 'bg-green-50', BORDER: 'border-l-4 border-green-500' }, 
            'New': { ICON: 'text-yellow-500', BG: 'bg-yellow-50', BORDER: 'border-l-4 border-yellow-500' }, 
            'Contacted': { ICON: 'text-yellow-500', BG: 'bg-yellow-50', BORDER: 'border-l-4 border-yellow-500' },
            'default': { ICON: 'text-gray-500', BG: 'bg-gray-50', BORDER: 'border-l-4 border-gray-500' }
        },
        ESTIMATE: {
            'sent': 'bg-[var(--color-primary)] text-white',
            'generated': 'bg-yellow-100 text-yellow-600'
        }
    };

    // 2. HELPERS
    const decodeHtmlEntities = (text) => {
        if (!text) return "";
        const textArea = document.createElement('textarea');
        textArea.innerHTML = text;
        return textArea.value;
    };
    const getIDValue = (element) => {
        const value = element.value;
        if (!value || value.trim() === '' || value.trim() === '0') { return null; }
        return value.trim();
    };
    const safeAddListener = (element, eventType, handler) => {
        if (element) element.addEventListener(eventType, handler);
    };
    const translateLeadStatus = (status) => {
        switch(status) {
            case 'New': return 'Nuevo'; case 'Contacted': return 'Contactado'; 
            case 'Qualified': return 'Calificado'; case 'Proposal Sent': return 'Cotización Enviada'; 
            case 'Negotiation': return 'Negociación Final'; default: return status;
        }
    };
    const translateQuoteStatus = (status) => {
        switch(status) {
            case 'generated': return 'Generado'; case 'sent': return 'Enviado'; default: return status;
        }
    };
    
    // 3. FUNCIONES DE CONTROL DE PANELES/MODALES
    const openPanel = (panel, overlay) => {
        if(!panel || !overlay) return;
        panel.classList.remove('translate-x-full');
        overlay.classList.remove('hidden');
        document.body.classList.add('overflow-hidden');
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };
    const closePanel = (panel, overlay) => {
        if(!panel || !overlay) return;
        panel.classList.add('translate-x-full');
        overlay.classList.add('hidden');
        if (DOMElements.confirmDeleteModal?.classList.contains('hidden') && DOMElements.toolsModal?.classList.contains('hidden')) {
            document.body.classList.remove('overflow-hidden');
        }
    };
    const closeModal = (modalId) => {
        const modal = document.getElementById(modalId);
        if (!modal) return;
        const modalBox = modal.querySelector('div:first-of-type');
        modalBox.classList.add('scale-95', 'opacity-0');
        setTimeout(() => {
            modal.classList.add('hidden');
            if (DOMElements.taskPanel?.classList.contains('translate-x-full') && DOMElements.editTaskPanel?.classList.contains('translate-x-full') && DOMElements.toolsModal?.classList.contains('hidden')) {
                document.body.classList.remove('overflow-hidden');
            }
        }, 300);
    };
    const openToolsModal = () => {
         const modal = DOMElements.toolsModal;
         if (!modal) return;
         const modalBox = DOMElements.toolsModalBox; 
         modal.classList.remove('hidden');
         document.body.classList.add('overflow-hidden');
         setTimeout(() => { modalBox.classList.remove('scale-95', 'opacity-0'); }, 50);
         if (typeof lucide !== 'undefined') lucide.createIcons();
    };
    const closeToolsModal = () => {
        const modal = DOMElements.toolsModal;
        if (!modal) return;
        const modalBox = DOMElements.toolsModalBox;
        modalBox.classList.add('scale-95', 'opacity-0');
        setTimeout(() => {
            modal.classList.add('hidden');
            if (DOMElements.taskPanel?.classList.contains('translate-x-full') && DOMElements.editTaskPanel?.classList.contains('translate-x-full') && DOMElements.confirmDeleteModal?.classList.contains('hidden')) {
                document.body.classList.remove('overflow-hidden');
            }
        }, 300);
    };
    
    // 4. FUNCIONES DE RENDERIZADO
        
    const createTaskListItem = (tarea) => {
        const template = document.getElementById('task-template');
        if (!template) return document.createElement('li');
        
        const listItem = template.content.cloneNode(true).querySelector('.task-list-item');
        if (!listItem) return document.createElement('li');
        
        const dueDate = new Date(`${tarea.due_date}T00:00:00`); 
        const today = new Date(); 
        today.setHours(0, 0, 0, 0);

        // Lógica de Estado y Mapeo (Usa STYLE_MAP)
        let style = STYLE_MAP.TASK.DEFAULT;
        let showDate = true;
        
        if (tarea.status === 'completed') { 
            style = STYLE_MAP.TASK.COMPLETED;
            showDate = false;
        } 
        else if (dueDate < today) { 
            const dateText = dueDate.toLocaleDateString('es-US', { day: 'numeric', month: 'short' });
            style = STYLE_MAP.TASK.OVERDUE(dateText);
            showDate = false;
        } 
        else if (dueDate.toDateString() === today.toDateString()) { 
            const dateText = dueDate.toLocaleDateString('es-US', { day: 'numeric', month: 'short' });
            style = STYLE_MAP.TASK.TODAY(dateText);
            showDate = false;
        }
        
        // Aplicación de Estilos y Clases 
        listItem.classList.remove('bg-white', 'border-l-4', 'border-[var(--color-primary)]'); 
        listItem.classList.add(style.BG, style.BORDER);
        listItem.querySelector('.task-title-text').classList.add(style.TITLE);
        listItem.querySelector('.task-status-badge').innerHTML = style.BADGE;
        
        // Inyección de Datos
        listItem.querySelector('[data-task-field="title"]').textContent = decodeHtmlEntities(tarea.title || 'Sin Título');
        listItem.querySelector('[data-task-field="priority"]').textContent = tarea.priority;

        const contactName = decodeHtmlEntities(tarea.client_name || tarea.prospect_name);
        const contactEl = listItem.querySelector('.task-contact-info');
        if (contactName) {
            contactEl.querySelector('[data-task-field="contact-name"]').textContent = contactName;
            contactEl.classList.remove('hidden');
        }

        const descriptionEl = listItem.querySelector('.task-description-text');
        if (tarea.description) {
            descriptionEl.querySelector('[data-task-field="description"]').textContent = decodeHtmlEntities(tarea.description);
            descriptionEl.classList.remove('hidden');
        }

        const dueDateContainer = listItem.querySelector('[data-task-field="due-date-container"]');
        if (showDate) {
            dueDateContainer.querySelector('[data-task-field="due-date"]').textContent = dueDate.toLocaleDateString('es-US', { day: 'numeric', month: 'short' });
            dueDateContainer.classList.remove('hidden');
        }

        const priorityBadgeEl = listItem.querySelector('.task-priority-badge');
        priorityBadgeEl.classList.add(STYLE_MAP.TASK.PRIORITY[tarea.priority] || STYLE_MAP.TASK.PRIORITY['Media']);

        // Inyección de Botones (Clonación de plantillas de botones)
        const actionContainer = listItem.querySelector('[data-task-actions]');
        let buttonsTemplate;
        
        if (tarea.status === 'pending') {
            buttonsTemplate = document.getElementById('task-buttons-pending');
        } else {
            buttonsTemplate = document.getElementById('task-buttons-completed');
        }
        
        if (buttonsTemplate) {
            actionContainer.appendChild(buttonsTemplate.content.cloneNode(true));
        }

        const deleteTemplate = document.getElementById('task-button-delete');
        if (deleteTemplate) {
            actionContainer.appendChild(deleteTemplate.content.cloneNode(true));
        }

        actionContainer.querySelectorAll('button').forEach(btn => {
            btn.dataset.taskId = tarea.id;
        });

        if (typeof lucide !== 'undefined') lucide.createIcons({ element: listItem });

        return listItem;
    };


    const updateSummaryCards = () => { 
        if(DOMElements.statTrackingProspects) DOMElements.statTrackingProspects.textContent = APP_DATA.summaryStats.trackingProspects?.toString() || '0'; 
        if(DOMElements.statPendingTasks) DOMElements.statPendingTasks.textContent = APP_DATA.summaryStats.pendingTasks?.toString() || '0';
        if(DOMElements.statPendingEstimates) DOMElements.statPendingEstimates.textContent = APP_DATA.summaryStats.pendingEstimates?.toString() || '0'; 
        if (typeof lucide !== 'undefined') { lucide.createIcons(); }
    };
    
    const renderUpcomingAppointments = () => { 
        const list = DOMElements.upcomingAppointmentsList; if (!list) return; list.innerHTML = '';
        const today = new Date(); today.setHours(0, 0, 0, 0);
        const filteredAppointments = APP_DATA.upcomingAppointments.sort((a, b) => new Date(`${a.fecha}T${a.hora}`) - new Date(`${b.fecha}T${b.hora}`));
        
        if (filteredAppointments.length === 0) { list.innerHTML = `<li class="text-base text-gray-500 uppercase">No hay citas próximas</li>`; } else {
            const fragment = document.createDocumentFragment();
            filteredAppointments.slice(0, 5).forEach(cita => {
                const listItem = document.createElement('li'); 
                const citaDate = new Date(`${cita.fecha}T${cita.hora}`); 
                const isToday = citaDate.toDateString() === today.toDateString();
                const borderColor = isToday ? 'border-l-4 border-[var(--color-highlight)]' : 'border-l-4 border-gray-300';
                const todayBadge = isToday ? `<span class="ml-2 text-xs font-bold uppercase bg-[var(--color-highlight)] text-[var(--color-primary)] px-2 py-0.5 rounded-full">HOY</span>` : '';
                
                listItem.className = `flex items-center space-x-3 p-2 rounded-lg hover:bg-gray-100 transition-all duration-300 shadow-sm ${borderColor} bg-white`;
                listItem.innerHTML = `
                    <div class="flex-shrink-0">
                        <div class="flex flex-col items-center justify-center bg-gray-100 rounded-lg p-1.5 w-12 h-12 border"> 
                            <span class="text-lg font-black text-[var(--color-secondary)]">${citaDate.toLocaleDateString('es-ES', { day: 'numeric' })}</span> 
                            <span class="text-xs font-bold text-gray-500 uppercase">${citaDate.toLocaleDateString('es-ES', { month: 'short' })}</span>
                        </div>
                    </div>
                    <div class="flex-1 min-w-0">
                        <p class="font-bold text-sm text-gray-800 uppercase truncate">${decodeHtmlEntities(cita.associated_name) || 'Cita sin asignar'}</p> 
                        <p class="text-xs text-gray-600 font-semibold uppercase truncate">${decodeHtmlEntities(cita.servicio) || 'Servicio no especificado'}</p> 
                        <p class="text-xs text-gray-500 flex items-center mt-0.5">
                            <i data-lucide="clock" class="w-3 h-3 mr-1"></i> 
                            ${new Date(`1970-01-01T${cita.hora}`).toLocaleTimeString('es-ES', { hour: '2-digit', minute: '2-digit', hour12: true })}${todayBadge}
                        </p>
                    </div>
                    <div class="flex-shrink-0">
                        <button class="text-gray-400 hover:text-[var(--color-primary)] transition-colors duration-200 go-to-agenda" title="Ir a la Agenda">
                            <i data-lucide="arrow-right-circle" class="w-5 h-5"></i> 
                        </button>
                    </div>
                `;
                fragment.appendChild(listItem);
            });
            list.appendChild(fragment);
        }
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };

    const renderProspectos = () => {
        const list = DOMElements.listaProspectosSeguimiento; if (!list) return; list.innerHTML = '';
        const trackingStatuses = ['New', 'Contacted', 'Qualified', 'Proposal Sent', 'Negotiation']; 
        const filteredProspects = APP_DATA.trackingProspectsList.filter(p => trackingStatuses.includes(p.estado)).sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
        
        if (filteredProspects.length === 0) { 
            list.innerHTML = '<li class="text-base text-gray-500 uppercase">No Hay Prospectos En Seguimiento Activos</li>'; 
            return; 
        }
        
        const fragment = document.createDocumentFragment();
        filteredProspects.slice(0, 5).forEach(prospecto => {
            const prospectoCard = document.createElement('div');
            const state = STYLE_MAP.LEAD[prospecto.estado] || STYLE_MAP.LEAD.default; 

            prospectoCard.className = `p-3 rounded-lg shadow-sm border flex items-center space-x-3 transition-all duration-200 ${state.BORDER} ${state.BG}`; 
            
            const fullName = `${prospecto.first_name || ''} ${prospecto.last_name || ''}`.trim() || prospecto.email || prospecto.phone;
            const translatedStatus = translateLeadStatus(prospecto.estado || prospecto.status_seguimiento);

            prospectoCard.innerHTML = `
                <div class="flex-shrink-0 p-2 rounded-lg bg-opacity-20 ${state.ICON} ${state.BG}">
                    <i data-lucide="user" class="w-5 h-5 ${state.ICON}"></i> 
                </div>
                <div class="flex-1 min-w-0">
                    <p class="font-bold text-sm text-gray-800 uppercase truncate">${decodeHtmlEntities(fullName)}</p> 
                    <p class="text-xs text-gray-600">Estado: <span class="font-semibold capitalize">${translatedStatus}</span></p> 
                </div>
                <button class="text-gray-500 hover:text-[var(--color-secondary)] transition-colors duration-200 edit-prospect-btn" data-prospect-id="${prospecto.id}" title="Editar Prospecto">
                    <i data-lucide="edit" class="w-6 h-6"></i> 
                </button>
            `;
            fragment.appendChild(prospectoCard);
        });
        list.appendChild(fragment);
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };

    const renderPendingEstimates = () => {
        const list = DOMElements.pendingEstimatesList; if (!list) return; list.innerHTML = '';
        const filteredEstimates = APP_DATA.pendingEstimatesList.filter(e => ['generated', 'sent'].includes(e.status)).sort((a, b) => new Date(b.estimate_date) - new Date(a.estimate_date));
        
        if (filteredEstimates.length === 0) { list.innerHTML = '<li class="text-base text-gray-500 uppercase">No hay estimados pendientes de seguimiento</li>'; return; }
        const fragment = document.createDocumentFragment();
        
        filteredEstimates.slice(0, 5).forEach(estimado => {
            const item = document.createElement('li'); 
            const statusColor = STYLE_MAP.ESTIMATE[estimado.status] || STYLE_MAP.ESTIMATE.generated;
            const clientName = estimado.client_name || estimado.lead_name || 'Cliente Desconocido';
            
            item.className = 'flex items-center justify-between space-x-4 p-3 rounded-lg hover:bg-gray-100 transition-all duration-300 shadow-sm border-l-4 border-[var(--color-primary)] bg-white cursor-pointer go-to-estimates-btn';
            item.dataset.estimateId = estimado.id;
            
            item.innerHTML = `
                <div class="flex-1 min-w-0">
                    <p class="font-extrabold text-sm text-gray-800 uppercase truncate">${decodeHtmlEntities(clientName)}</p>
                    <p class="text-xs text-gray-600 font-semibold mt-1 flex items-center">
                        <i data-lucide="hash" class="w-3 h-3 mr-1.5 text-[var(--color-primary)]"></i> ${estimado.external_id || estimado.id}
                    </p>
                    <p class="text-xs text-gray-500 mt-0.5">Fecha: ${new Date(`${estimado.estimate_date}T00:00:00`).toLocaleDateString('es-ES', { day: 'numeric', month: 'short', year: 'numeric'})}</p>
                </div>
                <div class="flex-shrink-0 text-right">
                    <p class="text-lg font-black text-[var(--color-primary)]">$${parseFloat(estimado.total_amount).toFixed(2)}</p>
                    <span class="px-2 py-0.5 rounded-full text-xs font-bold uppercase ml-2 ${statusColor}">${translateQuoteStatus(decodeHtmlEntities(estimado.status))}</span>
                </div>
            `;
            fragment.appendChild(item);
        });
        list.appendChild(fragment);
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };

    const renderOverdueInvoices = () => { 
        const list = DOMElements.overdueInvoicesList; if (!list) return; list.innerHTML = ''; 
        const today = new Date(); today.setHours(0, 0, 0, 0);
        
        const filteredInvoices = APP_DATA.overdueInvoicesList.filter(f => f.status !== 'paid' && f.due_date).sort((a, b) => { 
            const dateA = a.due_date ? new Date(`${a.due_date}T00:00:00`) : new Date(0); 
            const dateB = b.due_date ? new Date(`${b.due_date}T00:00:00`) : new Date(0); 
            return dateA - dateB; 
        });
        
        if (filteredInvoices.length === 0) { list.innerHTML = '<li class="text-base text-gray-500 uppercase">¡Felicidades! No hay facturas vencidas por cobrar</li>'; return; }
        
        const fragment = document.createDocumentFragment();
        filteredInvoices.slice(0, 5).forEach(factura => {
            const item = document.createElement('li'); 
            const vencimientoDate = factura.due_date ? new Date(`${factura.due_date}T00:00:00`) : null; 
            
            let daysOverdueText = 'Vencida';
            let statusBadgeClasses = 'bg-[var(--color-secondary)] text-white'; 
            
            if (vencimientoDate && vencimientoDate < today) { 
                const diffTime = today.getTime() - vencimientoDate.getTime();
                const daysOverdue = Math.max(1, Math.floor(diffTime / (1000 * 60 * 60 * 24))); 
                daysOverdueText = `${daysOverdue} día${daysOverdue !== 1 ? 's' : ''} vencida`; 
                statusBadgeClasses = 'bg-[var(--color-secondary)] text-white';
            } else if (vencimientoDate && vencimientoDate.toDateString() === today.toDateString()) {
                daysOverdueText = 'VENCE HOY';
                statusBadgeClasses = 'bg-orange-500 text-white';
            } else if (vencimientoDate && vencimientoDate > today) {
                const diffTime = vencimientoDate.getTime() - today.getTime();
                const daysLeft = Math.max(1, Math.ceil(diffTime / (1000 * 60 * 60 * 24))); 
                daysOverdueText = `Faltan ${daysLeft} días`;
                statusBadgeClasses = 'bg-yellow-500 text-white';
            }

            const clientName = factura.client_name || factura.lead_name || 'Cliente Desconocido';
            
            item.className = 'flex items-center justify-between space-x-4 p-3 rounded-lg hover:bg-red-50 transition-all duration-300 shadow-sm border-l-4 border-[var(--color-secondary)] bg-white cursor-pointer go-to-invoices-btn';
            item.dataset.invoiceId = factura.id;
            
            item.innerHTML = `
                <div class="flex-1 min-w-0">
                    <p class="font-extrabold text-sm text-[var(--color-secondary)] uppercase truncate">${decodeHtmlEntities(clientName)}</p>
                    <p class="text-xs text-gray-600 font-semibold mt-1 flex items-center">
                        <i data-lucide="hash" class="w-3 h-3 mr-1.5 text-[var(--color-secondary)]"></i> ${factura.external_id || factura.id}
                    </p>
                    <p class="text-xs text-gray-500 mt-0.5">Vence: ${vencimientoDate ? vencimientoDate.toLocaleDateString('es-ES', { day: 'numeric', month: 'short', year: 'numeric'}) : 'N/A'}</p>
                </div>
                <div class="flex-shrink-0 text-right">
                    <p class="text-lg font-black text-[var(--color-secondary)]">$${parseFloat(factura.total_amount).toFixed(2)}</p>
                    <span class="px-2 py-0.5 rounded-full text-xs font-bold uppercase ${statusBadgeClasses}">${daysOverdueText}</span>
                </div>
            `;
            fragment.appendChild(item);
        });
        list.appendChild(fragment);
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };
        
    const renderTasks = () => {
        if(!DOMElements.pendingTasksListEl || !DOMElements.completedTasksListEl) return;
        DOMElements.pendingTasksListEl.innerHTML = '';  
        DOMElements.completedTasksListEl.innerHTML = '';
        
        const pendingTasks = APP_DATA.allTasks.filter(task => task.status === 'pending');
        const completedTasks = APP_DATA.allTasks.filter(task => task.status === 'completed');
        
        if (pendingTasks.length === 0) DOMElements.pendingTasksListEl.innerHTML = '<li class="text-base text-gray-500 uppercase">No tienes tareas pendientes. ¡Bien hecho!</li>';
        else {
            const todayTimestamp = new Date().setHours(0, 0, 0, 0);  
            const priorityOrder = { 'Alta': 1, 'Media': 2, 'Baja': 3 };
            
            const sortedPendingTasks = pendingTasks
                .map(task => {  
                    const dueTimestamp = new Date(task.due_date).setHours(0, 0, 0, 0);  
                    return {  
                        ...task,  
                        _isOverdue: dueTimestamp < todayTimestamp,  
                        _isToday: dueTimestamp === todayTimestamp,  
                        _priority: priorityOrder[task.priority] || 4,  
                        _dueDate: dueTimestamp  
                    };  
                })
                .sort((a, b) =>  
                    (b._isOverdue - a._isOverdue) ||  
                    (b._isToday - a._isToday) ||    
                    (a._priority - b._priority) ||    
                    (a._dueDate - b._dueDate)      
                );
                
            const fragment = document.createDocumentFragment();  
            sortedPendingTasks.forEach(tarea => fragment.appendChild(createTaskListItem(tarea)));  
            DOMElements.pendingTasksListEl.appendChild(fragment);
        }
        
        if (completedTasks.length === 0) DOMElements.completedTasksListEl.innerHTML = '<li class="text-base text-gray-500 uppercase">Aún no hay tareas completadas</li>';
        else {
            const sortedCompletedTasks = completedTasks.sort((a, b) => new Date(b.updated_at) - new Date(a.updated_at));  
            const fragment = document.createDocumentFragment();
            sortedCompletedTasks.slice(0, 5).forEach(tarea => fragment.appendChild(createTaskListItem(tarea)));  
            DOMElements.completedTasksListEl.appendChild(fragment);
        }
        
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };

    // 5. FUNCIONES DE API (Usan CONFIG)
    const fetchContacts = async (query, type) => {
        if (!query || query.length < 2) return [];
        try {
            const response = await fetch(`${CONFIG.API_CONTACT_SEARCH_URL}?q=${encodeURIComponent(query)}&type=${type}`);
            const result = await response.json();
            return result.success ? result.data : [];
        } catch (error) {
            console.error("Error fetching contacts:", error);
            if (typeof showToast === 'function') showToast('Error al buscar contactos.', 'error');
            return [];
        }
    };
    
    const saveTask = async (formData, isEdit = false) => {
        const url = isEdit ? CONFIG.API_TASK_UPDATE_URL : CONFIG.API_TASK_CREATE_URL;
        formData.csrf_token = CONFIG.CSRF_TOKEN; 
        try {
            const response = await fetch(url, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(formData) });
            const result = await response.json();
            
            if (result.success) {
                if (typeof showToast === 'function') showToast(`Tarea ${isEdit ? 'actualizada' : 'agregada'} con éxito.`, 'success');
                isEdit ? closePanel(DOMElements.editTaskPanel, DOMElements.taskPanelOverlay) : closePanel(DOMElements.taskPanel, DOMElements.taskPanelOverlay);
                await initDashboard();
            } else { if (typeof showToast === 'function') showToast(`Error: ${result.message}`, 'error'); }
        } catch (error) { console.error("Save Task Error:", error); if (typeof showToast === 'function') showToast('Error de conexión con el servidor.', 'error'); }
    };
    
    const updateTaskStatus = async (taskId, newStatus) => {
        const formData = { id: taskId, status: newStatus, csrf_token: CONFIG.CSRF_TOKEN };
        try {
            const response = await fetch(CONFIG.API_TASK_UPDATE_URL, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(formData) });
            const result = await response.json();
            
            if (result.success) { if (typeof showToast === 'function') showToast('Estado de la tarea actualizado con éxito.', 'success'); await initDashboard(); } 
            else { if (typeof showToast === 'function') showToast(`Error: ${result.message}`, 'error'); }
        } catch (error) { console.error("Update Status Error:", error); if (typeof showToast === 'function') showToast('Error de conexión.', 'error'); }
    };

    const deleteTask = async (taskId) => {
        const formData = { id: taskId, csrf_token: CONFIG.CSRF_TOKEN };
        try {
            const response = await fetch(CONFIG.API_TASK_DELETE_URL, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(formData) });
            const result = await response.json();
            
            if (result.success) { if (typeof showToast === 'function') showToast('Tarea eliminada con éxito.', 'success'); await initDashboard(); } 
            else { if (typeof showToast === 'function') showToast(`Error: ${result.message}`, 'error'); }
        } catch (error) { console.error("Delete Task Error:", error); if (typeof showToast === 'function') showToast('Error de conexión.', 'error'); }
    };

    // 6. LÓGICA DE AUTOCOMPLETADO Y DASHBOARD
    const setupAutocomplete = (inputElement, hiddenIdElement, suggestionsDiv, type, otherInput) => {
        if(!inputElement || !suggestionsDiv) return;
        
        const clearSuggestions = () => {
            suggestionsDiv.textContent = '';
            suggestionsDiv.classList.add('hidden');
        };

        inputElement.addEventListener('input', function() {
            const searchTerm = this.value.trim();
            clearSuggestions();
            hiddenIdElement.value = '';

            if(otherInput) otherInput.disabled = searchTerm.length > 0; 
            
            if (searchTerm.length < 2) return;

            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(async () => {
                const results = await fetchContacts(searchTerm, type);
                
                if (results.length === 0) {
                    clearSuggestions();
                    return;
                }

                suggestionsDiv.classList.remove('hidden');
                
                results.forEach(item => {
                    const suggestionItem = document.createElement('div');
                    suggestionItem.className = 'px-3 py-2 cursor-pointer border-b border-gray-100 hover:bg-gray-100 text-sm text-gray-700 last:border-0 flex flex-col';
                    suggestionItem.innerHTML = `<span class="font-bold">${decodeHtmlEntities(item.label)}</span><span class="text-xs text-gray-500">${decodeHtmlEntities(item.sub_text)}</span>`;
                    
                    suggestionItem.addEventListener('click', function() {
                        inputElement.value = decodeHtmlEntities(item.label);
                        hiddenIdElement.value = item.id;
                        clearSuggestions();
                        
                        if(otherInput) {
                            otherInput.value = '';
                            const isAddPanel = inputElement.id.includes('add-');
                            const otherHiddenId = (type === 'clients') ? 
                                (isAddPanel ? DOMElements.addTaskProspectIdInput : DOMElements.editTaskProspectIdInput) : 
                                (isAddPanel ? DOMElements.addTaskClientIdInput : DOMElements.editTaskClientIdInput);
                                
                            otherHiddenId.value = '';
                            otherInput.disabled = true;
                        }
                    });
                    suggestionsDiv.appendChild(suggestionItem);
                });
            }, 300); 
        });

        inputElement.addEventListener('keyup', function() {
            if (this.value.trim().length === 0) {
                hiddenIdElement.value = '';
                if (otherInput) otherInput.disabled = false;
                clearSuggestions();
            }
        });

        inputElement.addEventListener('blur', () => setTimeout(clearSuggestions, 150));
    };

    const initDashboard = async () => {
        try {
            const cacheBust = new Date().getTime();
            const dashboardResponse = await fetch(`${CONFIG.API_DASHBOARD_READ_URL}?v=${cacheBust}`);
            const dashboardResult = await dashboardResponse.json();
            
            if (dashboardResult.success && dashboardResult.data) {
                APP_DATA.summaryStats = { ...dashboardResult.data };  
                APP_DATA.upcomingAppointments = dashboardResult.data.upcomingAppointments || [];  
                APP_DATA.allTasks = dashboardResult.data.allTasks || [];  
                APP_DATA.trackingProspectsList = dashboardResult.data.trackingProspectsList || [];  
                APP_DATA.pendingEstimatesList = dashboardResult.data.pendingEstimatesList || [];  
                APP_DATA.overdueInvoicesList = dashboardResult.data.overdueInvoicesList || [];
            }
        } catch (error) {  
            console.error(error);  
            if (typeof showToast === 'function') showToast('Error al cargar datos del dashboard.', 'error');  
        }
        
        updateSummaryCards();  
        renderUpcomingAppointments();  
        renderProspectos();  
        renderPendingEstimates();  
        renderOverdueInvoices();  
        renderTasks();
        
        setupAutocomplete(DOMElements.addTaskClientNameInput, DOMElements.addTaskClientIdInput, DOMElements.addClientSuggestionsDiv, 'clients', DOMElements.addTaskProspectNameInput);
        setupAutocomplete(DOMElements.addTaskProspectNameInput, DOMElements.addTaskProspectIdInput, DOMElements.addProspectSuggestionsDiv, 'leads', DOMElements.addTaskClientNameInput);
        
        setupAutocomplete(DOMElements.editTaskClientNameInput, DOMElements.editTaskClientIdInput, DOMElements.editClientSuggestionsDiv, 'clients', DOMElements.editTaskProspectNameInput);
        setupAutocomplete(DOMElements.editTaskProspectNameInput, DOMElements.editTaskProspectIdInput, DOMElements.editProspectSuggestionsDiv, 'leads', DOMElements.editTaskClientNameInput);

        const isEditClientSet = !!DOMElements.editTaskClientNameInput?.value;
        const isEditLeadSet = !!DOMElements.editTaskProspectNameInput?.value;

        if (DOMElements.editTaskClientNameInput) DOMElements.editTaskClientNameInput.disabled = isEditLeadSet;
        if (DOMElements.editTaskProspectNameInput) DOMElements.editTaskProspectNameInput.disabled = isEditClientSet;
    };
    
    // 7. MANEJADORES DE EVENTOS DE PANELES/MODALES
    const handleListClick = (e) => {
        const target = e.target;
        const editBtn = target.closest('.edit-task-btn');
        const completeBtn = target.closest('.complete-task-btn');
        const uncompleteBtn = target.closest('.uncomplete-task-btn');
        const deleteBtn = target.closest('.delete-task-btn');
        const editProspectBtn = target.closest('.edit-prospect-btn');
        const agendaBtn = target.closest('.go-to-agenda');
        const estimateBtn = target.closest('.go-to-estimates-btn');  
        const invoiceBtn = target.closest('.go-to-invoices-btn');  
        
        if (editBtn) openEditTaskPanel(editBtn.dataset.taskId);
        else if (completeBtn) updateTaskStatus(parseInt(completeBtn.dataset.taskId, 10), 'completed');
        else if (uncompleteBtn) updateTaskStatus(parseInt(uncompleteBtn.dataset.taskId, 10), 'pending');
        else if (deleteBtn) openConfirmDeleteModal(deleteBtn.dataset.taskId, 'tarea');
        else if (editProspectBtn) window.location.href = 'leads.php';
        else if (agendaBtn) window.location.href = 'agenda.php';
        else if (estimateBtn) window.location.href = 'quotes.php';  
        else if (invoiceBtn) window.location.href = 'invoices.php';  
    };
    
    const handleConfirmDelete = async () => {
        const itemId = DOMElements.confirmDeleteButton.dataset.itemId;
        const itemType = DOMElements.confirmDeleteButton.dataset.itemType;
        if (!itemId || !itemType) return;
        
        if (itemType === 'task') await deleteTask(parseInt(itemId, 10));
        
        closeModal('confirmDeleteModal');
    };
    
    const openConfirmDeleteModal = (itemId, itemType) => {
        let itemName = '';
        if (itemType === 'tarea') {
            const task = APP_DATA.allTasks.find(t => t.id == itemId);
            if (task) itemName = `"${decodeHtmlEntities(task.title)}"`; 
        }
        
        const nameSpan = document.getElementById('delete-item-name');
        if(nameSpan) nameSpan.textContent = itemName;
        
        DOMElements.confirmDeleteButton.dataset.itemId = itemId;
        DOMElements.confirmDeleteButton.dataset.itemType = itemType;
        
        const modal = DOMElements.confirmDeleteModal;
        const modalBox = DOMElements.confirmModalBox; 
        
        modal.classList.remove('hidden');
        setTimeout(() => modalBox.classList.remove('scale-95', 'opacity-0'), 50);
        
        if(DOMElements.taskPanelOverlay) DOMElements.taskPanelOverlay.classList.remove('hidden');
        document.body.classList.add('overflow-hidden');
        if (typeof lucide !== 'undefined') lucide.createIcons();
    };
    
    const openTaskPanel = () => {
        if(DOMElements.addTaskForm) DOMElements.addTaskForm.reset();
        
        DOMElements.addTaskClientIdInput.value = '';
        DOMElements.addTaskProspectIdInput.value = '';
        DOMElements.addClientSuggestionsDiv.innerHTML = '';  
        DOMElements.addClientSuggestionsDiv.classList.add('hidden');
        DOMElements.addProspectSuggestionsDiv.innerHTML = '';  
        DOMElements.addProspectSuggestionsDiv.classList.add('hidden');
        
        DOMElements.addTaskDueDateInput.value = new Date().toISOString().split('T')[0]; 
        DOMElements.addTaskPrioritySelect.value = 'Media';
        
        DOMElements.addTaskClientNameInput.disabled = false;
        DOMElements.addTaskProspectNameInput.disabled = false;
        
        DOMElements.addTaskCsrfTokenInput.value = CONFIG.CSRF_TOKEN;
        
        openPanel(DOMElements.taskPanel, DOMElements.taskPanelOverlay);
    };

    const openEditTaskPanel = (taskId) => {
        const task = APP_DATA.allTasks.find(t => t.id == taskId);
        if (!task) { if (typeof showToast === 'function') showToast("Tarea no encontrada.", 'error'); return; }
        
        DOMElements.editClientSuggestionsDiv.innerHTML = ''; DOMElements.editClientSuggestionsDiv.classList.add('hidden');
        DOMElements.editProspectSuggestionsDiv.innerHTML = ''; DOMElements.editProspectSuggestionsDiv.classList.add('hidden');
        
        DOMElements.editTaskIdInput.value = task.id;
        DOMElements.editTaskTitleInput.value = decodeHtmlEntities(task.title || '');
        DOMElements.editTaskDescriptionInput.value = decodeHtmlEntities(task.description || '');
        DOMElements.editTaskDueDateInput.value = task.due_date || '';
        DOMElements.editTaskPrioritySelect.value = task.priority || 'Media';
        
        DOMElements.editTaskClientIdInput.value = task.client_id || '';
        DOMElements.editTaskClientNameInput.value = decodeHtmlEntities(task.client_name || '');
        DOMElements.editTaskProspectIdInput.value = task.lead_id || '';
        DOMElements.editTaskProspectNameInput.value = decodeHtmlEntities(task.prospect_name || '');
        
        const isClientSet = !!task.client_id;
        const isLeadSet = !!task.lead_id;

        DOMElements.editTaskClientNameInput.disabled = isLeadSet;
        DOMElements.editTaskProspectNameInput.disabled = isClientSet;
        
        DOMElements.editTaskCsrfTokenInput.value = CONFIG.CSRF_TOKEN;
        
        openPanel(DOMElements.editTaskPanel, DOMElements.taskPanelOverlay);
    };

    const videoWidgetLogic = () => {
        const videoSection = document.getElementById('videotutorial-section');
        const playOverlay = document.getElementById('playOverlay');
        const videoPlayer = document.getElementById('videoPlayer');
        const showVideoBtn = document.getElementById('show-video-btn');
        const hideVideoBtn = document.getElementById('hide-video-btn');
        const container = document.getElementById('videotutorial-container');

        const showVideoWidget = () => {
            videoSection.classList.remove('hidden');
            container.classList.add('hidden');
            localStorage.setItem('showVideoWidget', 'true');
            if (typeof lucide !== 'undefined') lucide.createIcons();
        };

        const hideVideoWidget = () => {
            videoSection.classList.add('hidden');
            container.classList.remove('hidden');
            localStorage.setItem('showVideoWidget', 'false');
            if (videoPlayer) {
                videoPlayer.pause();
                if (playOverlay) playOverlay.classList.remove('hidden');
            }
        };

        if (localStorage.getItem('showVideoWidget') === 'true') {
            showVideoWidget();
        } else {
            hideVideoWidget();
        }

        safeAddListener(showVideoBtn, 'click', showVideoWidget);
        safeAddListener(hideVideoBtn, 'click', hideVideoWidget);
        
        if (playOverlay && videoPlayer) {
            safeAddListener(playOverlay, 'click', function() {
                videoPlayer.play();
                playOverlay.classList.add('hidden');
            });
            safeAddListener(videoPlayer, 'play', function() { playOverlay.classList.add('hidden'); });
            safeAddListener(videoPlayer, 'ended', function() { playOverlay.classList.remove('hidden'); videoPlayer.load(); });
        }
    };
    
    // 8. INICIALIZACIÓN PRINCIPAL
    document.addEventListener('DOMContentLoaded', () => {
        // Carga la configuración estática
        loadApiConstants(); 

        // Asignación de elementos DOM
        DOMElements = {
            mobileMenuButton: document.getElementById('mobile-menu-button'),
            sidebar: document.getElementById('sidebar'), 
            sidebarOverlay: document.getElementById('sidebar-overlay'),
            statTrackingProspects: document.getElementById('stat-tracking-prospects'),
            statPendingTasks: document.getElementById('stat-pending-tasks'),
            statPendingEstimates: document.getElementById('stat-pending-estimates'),  
            upcomingAppointmentsList: document.getElementById('upcoming-appointments-list'),
            listaProspectosSeguimiento: document.getElementById('lista-prospectos-seguimiento'),
            pendingEstimatesList: document.getElementById('pending-estimates-list'),  
            overdueInvoicesList: document.getElementById('overdue-invoices-list'),
            viewAllEstimatesButton: document.getElementById('view-all-estimates-button'),
            viewAllInvoicesButton: document.getElementById('view-all-invoices-button'),
            pendingTasksListEl: document.getElementById('pending-tasks-list'),
            completedTasksListEl: document.getElementById('completed-tasks-list'),
            addAppointmentButton: document.getElementById('add-appointment-button'),
            addProspectButton: document.getElementById('add-prospect-button'),
            addTaskButton: document.getElementById('add-task-button'),
            taskPanel: document.getElementById('task-panel'),
            taskPanelOverlay: document.getElementById('task-panel-overlay'),
            closeTaskPanelButton: document.getElementById('close-task-panel'),
            panelCancelButton: document.getElementById('panel-cancel-btn'),
            addTaskForm: document.getElementById('add-task-form'),
            editTaskPanel: document.getElementById('edit-task-panel'),
            closeEditTaskPanelButton: document.getElementById('close-edit-task-panel'),
            editPanelCancelButton: document.getElementById('edit-panel-cancel-btn'),
            editTaskForm: document.getElementById('edit-task-form'),
            addTaskTitleInput: document.getElementById('add-task-title'),
            addTaskClientNameInput: document.getElementById('add-task-client-name'),
            addTaskClientIdInput: document.getElementById('add-task-client-id'),
            addTaskProspectNameInput: document.getElementById('add-task-prospect-name'),
            addTaskProspectIdInput: document.getElementById('add-task-prospect-id'),
            addTaskDescriptionInput: document.getElementById('add-task-description'),
            addTaskDueDateInput: document.getElementById('add-task-due-date'),
            addTaskPrioritySelect: document.getElementById('add-task-priority'),
            addClientSuggestionsDiv: document.getElementById('add-client-suggestions'),
            addProspectSuggestionsDiv: document.getElementById('add-prospect-suggestions'),
            editTaskIdInput: document.getElementById('edit-task-id'),
            editTaskTitleInput: document.getElementById('edit-task-title'),
            editTaskClientNameInput: document.getElementById('edit-task-client-name'),
            editTaskClientIdInput: document.getElementById('edit-task-client-id'),
            editTaskProspectNameInput: document.getElementById('edit-task-prospect-name'),
            editTaskProspectIdInput: document.getElementById('edit-task-prospect-id'),
            editTaskDescriptionInput: document.getElementById('edit-task-description'),
            editTaskDueDateInput: document.getElementById('edit-task-due-date'),
            editTaskPrioritySelect: document.getElementById('edit-task-priority'),
            editClientSuggestionsDiv: document.getElementById('edit-client-suggestions'),
            editProspectSuggestionsDiv: document.getElementById('edit-prospect-suggestions'),
            confirmDeleteModal: document.getElementById('confirmDeleteModal'),
            confirmModalBox: document.getElementById('confirm-modal-box'),
            confirmDeleteButton: document.getElementById('confirm-delete-button'),
            cancelDeleteButton: document.getElementById('cancel-delete-button'),
            addTaskCsrfTokenInput: document.getElementById('add-task-csrf-token'),
            editTaskCsrfTokenInput: document.getElementById('edit-task-csrf-token'),
            floatingToolsBtn: document.getElementById('floating-tools-btn'),
            toolsModal: document.getElementById('toolsModal'),
            toolsModalBox: document.getElementById('tools-modal-box'),
            closeToolsModalBtn: document.getElementById('closeToolsModalBtn')
        };
        
        // 8. ASIGNACIÓN DE EVENTOS (Todos usan addEventListener)
        
        // Sidebar y Navegación
        safeAddListener(DOMElements.mobileMenuButton, 'click', () => { DOMElements.sidebar?.classList.toggle('-translate-x-full'); DOMElements.sidebarOverlay?.classList.toggle('hidden'); });
        safeAddListener(DOMElements.sidebarOverlay, 'click', () => { DOMElements.sidebar?.classList.add('-translate-x-full'); DOMElements.sidebarOverlay?.classList.add('hidden'); });
        safeAddListener(DOMElements.addAppointmentButton, 'click', () => { window.location.href = "agenda.php"; });
        safeAddListener(DOMElements.addProspectButton, 'click', () => { window.location.href = "leads.php"; });
        safeAddListener(DOMElements.viewAllEstimatesButton, 'click', () => { window.location.href = "quotes.php"; });
        safeAddListener(DOMElements.viewAllInvoicesButton, 'click', () => { window.location.href = "invoices.php"; });
        
        // Paneles de Tarea
        safeAddListener(DOMElements.addTaskButton, 'click', openTaskPanel);
        safeAddListener(DOMElements.closeTaskPanelButton, 'click', () => closePanel(DOMElements.taskPanel, DOMElements.taskPanelOverlay));
        safeAddListener(DOMElements.panelCancelButton, 'click', () => closePanel(DOMElements.taskPanel, DOMElements.taskPanelOverlay));
        safeAddListener(DOMElements.closeEditTaskPanelButton, 'click', () => closePanel(DOMElements.editTaskPanel, DOMElements.taskPanelOverlay));
        safeAddListener(DOMElements.editPanelCancelButton, 'click', () => closePanel(DOMElements.editTaskPanel, DOMElements.taskPanelOverlay));
        
        // Tools Modal
        safeAddListener(DOMElements.floatingToolsBtn, 'click', openToolsModal);
        safeAddListener(DOMElements.closeToolsModalBtn, 'click', closeToolsModal);
        
        // Envío de formularios (USA CONFIG)
        safeAddListener(DOMElements.addTaskForm, 'submit', (e) => {  
            e.preventDefault();  
            const taskData = {  
                title: DOMElements.addTaskTitleInput.value,  
                description: DOMElements.addTaskDescriptionInput.value,  
                client_id: getIDValue(DOMElements.addTaskClientIdInput),  
                client_name: DOMElements.addTaskClientNameInput.value || null,  
                lead_id: getIDValue(DOMElements.addTaskProspectIdInput),  
                lead_name: DOMElements.addTaskProspectNameInput.value || null,
                due_date: DOMElements.addTaskDueDateInput.value,  
                priority: DOMElements.addTaskPrioritySelect.value,  
                status: 'pending'  
            };  
            if (!taskData.title || !taskData.due_date) { if (typeof showToast === 'function') showToast('Completa los campos obligatorios.', 'warning'); return; }  
            saveTask(taskData, false);  
        });
        
        safeAddListener(DOMElements.editTaskForm, 'submit', (e) => {  
            e.preventDefault();  
            const taskData = {  
                id: DOMElements.editTaskIdInput.value,
                title: DOMElements.editTaskTitleInput.value,  
                description: DOMElements.editTaskDescriptionInput.value,  
                client_id: getIDValue(DOMElements.editTaskClientIdInput),  
                client_name: DOMElements.editTaskClientNameInput.value || null,  
                lead_id: getIDValue(DOMElements.editTaskProspectIdInput),  
                lead_name: DOMElements.editTaskProspectNameInput.value || null,  
                due_date: DOMElements.editTaskDueDateInput.value,  
                priority: DOMElements.editTaskPrioritySelect.value  
            };  
            if (!taskData.title || !taskData.due_date) { if (typeof showToast === 'function') showToast('Completa los campos obligatorios.', 'warning'); return; }  
            saveTask(taskData, true);  
        });
        
        // Overlays y Modales
        safeAddListener(DOMElements.taskPanelOverlay, 'click', () => { closePanel(DOMElements.taskPanel, DOMElements.taskPanelOverlay); closePanel(DOMElements.editTaskPanel, DOMElements.taskPanelOverlay); closeModal('confirmDeleteModal'); });
        safeAddListener(DOMElements.cancelDeleteButton, 'click', () => closeModal('confirmDeleteModal'));
        safeAddListener(DOMElements.confirmDeleteButton, 'click', handleConfirmDelete);
        
        // Delegación de eventos para listas
        safeAddListener(DOMElements.pendingTasksListEl, 'click', handleListClick);
        safeAddListener(DOMElements.completedTasksListEl, 'click', handleListClick);
        safeAddListener(DOMElements.listaProspectosSeguimiento, 'click', handleListClick);
        safeAddListener(DOMElements.upcomingAppointmentsList, 'click', handleListClick);
        safeAddListener(DOMElements.pendingEstimatesList, 'click', handleListClick);
        safeAddListener(DOMElements.overdueInvoicesList, 'click', handleListClick);
        
        // Lógica de video
        videoWidgetLogic();
        
        // Carga inicial de datos
        initDashboard();
    });
})();
</script>
</body>
</html>