<?php
ob_start();
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json');

try {
    // Verificación de API Key
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }
    $apiKey = GEMINI_API_KEY;

    // Verificación del método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido.']);
        exit;
    }

    // Obtención y validación de datos
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    // Sanitizar y validar las entradas
    $user_name = filter_var($data['user_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $company_name = filter_var($data['company_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $client_name = filter_var($data['client_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $service_type = filter_var($data['service_type'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $scenario = filter_var($data['scenario'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $message_tone = filter_var($data['message_tone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $additional_details = filter_var($data['additional_details'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $contact_phone = filter_var($data['contact_phone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $contact_web = filter_var($data['contact_web'] ?? '', FILTER_SANITIZE_URL);

    // Validar campos obligatorios
    if (empty($user_name) || empty($company_name) || empty($client_name) || empty($service_type) || empty($scenario) || empty($message_tone)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan campos obligatorios.']);
        exit;
    }

    // Mapeo de escenarios a descripciones para el prompt
    $scenario_descriptions = [
        'first_contact' => [
            'purpose' => "un mensaje de primer contacto para un prospecto. El objetivo es presentarte, confirmar su interés, y preguntar sobre sus necesidades para agendar una llamada. Debe ser breve y amigable.",
            'tip' => "El consejo debe enfocarse en la importancia de ser rápido y personalizado en el primer contacto. Sugiere terminar con una pregunta abierta para iniciar una conversación real y no un monólogo."
        ],
        'follow_up_1' => [
            'purpose' => "un mensaje de seguimiento #1 para un prospecto que no ha respondido al primer contacto (después de 1-2 días). Debe ser un recordatorio suave y preguntar si tiene alguna duda.",
            'tip' => "El consejo debe orientar sobre la cadencia del seguimiento. Recomienda ser persistente pero no intrusivo. Mantener un tono amigable, ofreciendo valor en cada mensaje."
        ],
        'follow_up_2' => [
            'purpose' => "un mensaje de seguimiento #2 para un prospecto que no ha respondido (después de 3-5 días). Debe ser una última oportunidad para conectar, ofreciendo algo de valor adicional (ej. un caso de éxito) antes de cerrar el seguimiento.",
            'tip' => "El consejo debe centrarse en cuándo detener el seguimiento. Recomienda un límite para no molestar y mantener una imagen profesional. Sugiere cerrar el ciclo de seguimiento de forma respetuosa."
        ],
        'quote_sent' => [
            'purpose' => "un mensaje para enviar una cotización formal. El mensaje debe ser profesional, adjuntar el documento de cotización y resumir los puntos clave para que el prospecto entienda rápidamente el valor y los beneficios.",
            'tip' => "El consejo debe centrarse en cómo presentar la cotización. Sugiere resaltar los beneficios principales de la propuesta en el mensaje. También, incluye un CTA claro para agendar una llamada."
        ],
        'follow_up_quote_1' => [
            'purpose' => "un mensaje de seguimiento #1 para una cotización que fue enviada (después de 1-2 días) y aún no ha tenido respuesta. Debe ser un recordatorio amable y preguntar si tienen alguna duda para avanzar en el proceso.",
            'tip' => "El consejo debe guiar en el manejo de cotizaciones no respondidas. Recomienda preguntar sobre el momento adecuado para el prospecto, en lugar de preguntar '¿qué te pareció?'. Propone un siguiente paso claro."
        ],
        'follow_up_quote_2' => [
            'purpose' => "un mensaje de seguimiento #2 para una cotización que no ha tenido respuesta (después de 3-5 días). Es la última oportunidad para conectar antes de cerrar el ciclo de seguimiento. Debe ser un mensaje directo y respetuoso, preguntando si hay un mejor momento para hablar o si sus prioridades han cambiado.",
            'tip' => "El consejo debe centrarse en la importancia de saber cuándo dejar ir a un prospecto de forma elegante. Esto mantiene una relación positiva en caso de que sus necesidades cambien en el futuro."
        ]
    ];

    $purpose = $scenario_descriptions[$scenario]['purpose'];
    $tip_purpose = $scenario_descriptions[$scenario]['tip'];

    // Construcción del CTA basado en los datos proporcionados
    $cta_block = "";
    if (!empty($contact_phone) && !empty($contact_web)) {
        $cta_block = "\n\n¡Estoy aquí para ayudarte a avanzar!\n📞 Contacto Rápido: {$contact_phone}\n🌐 Visita nuestra página: {$contact_web}\n";
    } elseif (!empty($contact_phone)) {
        $cta_block = "\n\n¡Estoy aquí para ayudarte a avanzar!\n📞 Contacto Rápido: {$contact_phone}\n";
    } elseif (!empty($contact_web)) {
        $cta_block = "\n\n¡Estoy aquí para ayudarte a avanzar!\n🌐 Visita nuestra página: {$contact_web}\n";
    }

    // Construcción del PROMPT MAESTRO con el nuevo CTA y el tono
    $prompt = "Actúa como un experto en redacción de mensajes de ventas para negocios de servicios locales. Tu objetivo es generar 3 ideas de mensajes diferentes, persuasivos y empáticos para guiar a los prospectos hacia la venta.

    **CONTEXTO CLAVE:**
    - Tu Nombre (el remitente): \"{$user_name}\"
    - Nombre de Tu Negocio: \"{$company_name}\"
    - Nombre del Prospecto (el destinatario): \"{$client_name}\"
    - Servicio de Interés: \"{$service_type}\"
    - Escenario: {$purpose}
    - Tono Solicitado: {$message_tone}
    " . (!empty($additional_details) ? "- Detalles Adicionales: \"{$additional_details}\"\n" : "") . "

    **TU TAREA:**
    Genera 3 sugerencias de mensajes diferentes, cada una con un enfoque estratégico ligeramente distinto, pero siempre manteniendo el tono solicitado. Para cada sugerencia, genera:
    1.  **spanish_message (Mensaje en Español):** Un mensaje conciso para el prospecto. Debe ser muy corto, ideal para WhatsApp o Messenger (máximo 5-7 líneas). Utiliza el nombre del prospecto y del negocio de forma natural. Debe terminar con un CTA claro que invite al siguiente paso. Incorpora el siguiente bloque de CTA al final del mensaje, solo si está disponible: {$cta_block} Utiliza saltos de línea (`\n`) para un formato legible.
    2.  **english_message (Mensaje en Inglés):** La traducción exacta del `spanish_message` al inglés, manteniendo el mismo tono y estructura. Asegúrate de que el CTA también esté traducido.
    3.  **tip (Consejo Estratégico):** Un consejo práctico para el dueño del negocio sobre la estrategia detrás de este tipo de mensaje. **Este consejo debe ser conciso, accionable y redactado ÚNICAMENTE en español**. {$tip_purpose}.

    **REGLAS IMPORTANTES:**
    -   **NO uses marcadores de posición** como `[Nombre del Prospecto]` en los mensajes finales.
    -   **NO incluyas texto introductorio o explicaciones** fuera del objeto JSON.
    -   La salida debe ser un **objeto JSON válido** que contenga un array llamado `suggestions`. Este array debe contener 3 objetos, cada uno con las claves \"spanish_message\", \"english_message\" y \"tip\".
    ";

    // Llamada a la API de Google Gemini con esquema forzado
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'suggestions' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'spanish_message' => ['type' => 'STRING'],
                                'english_message' => ['type' => 'STRING'],
                                'tip' => ['type' => 'STRING']
                            ],
                            'required' => ['spanish_message', 'english_message', 'tip']
                        ],
                        'minItems' => 3,
                        'maxItems' => 3
                    ]
                ],
                'required' => ['suggestions']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // Procesamiento y validación de la respuesta
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502);
        error_log("Error de conexión/respuesta de IA: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . $response_ai);
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA.', 'details' => $curl_error ?: 'No hay detalles de cURL.']);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;
    
    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado. Raw: " . $response_ai);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado.']);
        exit;
    }

    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['suggestions']) || !is_array($final_data['suggestions']) || count($final_data['suggestions']) !== 3) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido. Raw AI Text: " . trim($response_text));
        echo json_encode(['error' => 'La IA devolvió un formato de JSON inválido.']);
        exit;
    }

    // Envío de la respuesta final
    ob_end_clean();
    echo json_encode($final_data);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    while (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>