<?php
ob_start();
require_once __DIR__ . '/../config.php';

// CORRECCIÓN 1: Especificar la codificación UTF-8 en la cabecera.
header('Content-Type: application/json; charset=utf-8');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y DATOS
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada en el servidor. Verifica tu archivo config.php.']);
        exit;
    }
    if (!defined('GEMINI_MODEL') || empty(GEMINI_MODEL)) {
        define('GEMINI_MODEL', 'gemini-1.5-flash'); // Fallback si no está definida en config
    }

    $apiKey = GEMINI_API_KEY;

    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido. ' . json_last_error_msg()]);
        exit;
    }

    // Sanitizar y validar todas las entradas
    $business_name = filter_var($data['business_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $website_url = filter_var($data['website_url'] ?? '', FILTER_SANITIZE_URL);
    $phone_whatsapp = filter_var($data['phone_whatsapp'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $service_zone = filter_var($data['service_zone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $prospect_name = filter_var($data['prospect_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $product_service_description = filter_var($data['product_service_description'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $desired_outcome_client = filter_var($data['desired_outcome_client'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $contact_context = filter_var($data['contact_context'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $desired_outcome_message = filter_var($data['desired_outcome'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $communication_tone = filter_var($data['communication_tone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS); // Capturado correctamente

    $required_fields_check = [
        'business_name',
        'product_service_description',
        'desired_outcome_client',
        'contact_context',
        'desired_outcome_message',
        'communication_tone',
        'value_proposition'
    ];

    foreach ($required_fields_check as $field) {
        if (empty(${$field})) {
            http_response_code(400);
            echo json_encode(['error' => "Falta el campo requerido: '{$field}'. Asegúrate de completar todos los datos."]);
            exit;
        }
    }

    // --- PROMPT MAESTRO ---
    $prompt = "Actúa como un experto en ventas y comunicación para negocios locales. Tu tarea es generar opciones de mensajes de primer contacto altamente efectivos y persuasivos, adaptados al contexto y al prospecto, siempre con un llamado a la acción claro.

--- FILOSOFÍA CLIENT3 ---
- El primer contacto debe ser relevante y personalizado.
- Debe enfocarse en el prospecto, no solo en vender.
- El objetivo es iniciar una conversación, no cerrar la venta de inmediato.
- Debe ser conciso y fácil de entender.

--- TU TAREA ESPECÍFICA ---
Un usuario necesita mensajes para iniciar contacto con nuevos prospectos. Basado en la siguiente información, genera 3 opciones de mensajes de primer contacto. Cada mensaje debe tener su versión en español y en inglés.
**La IA DEBE INFERIR** el problema del cliente y los beneficios del servicio a partir de la 'Descripción del Servicio Principal' y el 'Deseo/Resultado Ideal del Cliente'.

--- DATOS PROPORCIONADOS ---
- Nombre del Negocio: \"" . $business_name . "\"
- Propuesta de Valor: \"" . $value_proposition . "\" // Añadido para el contexto estratégico
- Descripción del Servicio Principal: \"" . $product_service_description . "\"
- Deseo o Resultado Ideal del Cliente: \"" . $desired_outcome_client . "\"
- Contexto del Contacto (cómo los encontraste/por qué les escribes): \"" . $contact_context . "\"
- Resultado Deseado del Mensaje (qué quieres que hagan): \"" . $desired_outcome_message . "\"
- Tono de Comunicación Deseado: \"" . $communication_tone . "\"
";

if (!empty($prospect_name)) {
    $prompt .= "- Nombre Específico del Prospecto: \"" . $prospect_name . "\"\n";
}
if (!empty($website_url)) {
    $prompt .= "- Página Web/Perfil Social: \"" . $website_url . "\"\n";
}
if (!empty($phone_whatsapp)) {
    $prompt .= "- Teléfono/WhatsApp de Contacto: \"" . $phone_whatsapp . "\"\n";
}
if (!empty($service_zone)) {
    $prompt .= "- Zona Geográfica de Servicio: \"" . $service_zone . "\"\n";
}

$prompt .= "
--- REGLAS ESTRICTAS PARA LA GENERACIÓN ---
- **Output Bilingüe:** Para CADA mensaje, DEBES proporcionar una versión en 'es' (español) y 'en' (inglés).
- **Concisión:** Cada mensaje (es/en) debe ser conciso (máximo 5-7 líneas).
- **Personalización ESTRICTA:** DEBES usar directamente la información proporcionada en 'DATOS PROPORCIONADOS' para construir los mensajes.
- **PROHIBIDO Usar Marcadores de Posición `[...]`:** NUNCA incluyas texto entre corchetes `[...]` en el mensaje generado. SI se te proporcionó un 'Nombre Específico del Prospecto', DEBES usarlo directamente. Si no se te proporcionó, asume que el usuario lo insertará manualmente o usa un saludo general y profesional.
- **Enfoque en el Prospecto:** Empieza el mensaje centrándote en el problema o interés del prospecto, derivado de la descripción del servicio y el deseo del cliente.
- **Conecta Problema y Solución:** Asegúrate de que el mensaje conecte el problema del prospecto con TU servicio/producto y sus beneficios.
- **Llamado a la Acción (CTA):** Cada mensaje debe terminar con un CTA claro, alineado con el 'Resultado Deseado del Mensaje'. Incluye el teléfono/WhatsApp si fue proporcionado.
- **Tono:** El tono de la publicación debe ser coherente con el 'Tono de Comunicación Deseado'.
- **Emojis:** Usa emojis de forma profesional y moderada si encajan con el tono (ej. ✨, 🚀, ✅, ⭐).

--- FORMATO DE RESPUESTA REQUERIDO (JSON) ---
La respuesta DEBE ser un objeto JSON válido con una clave 'messages', que es un array de 3 objetos. Cada objeto en 'messages' debe tener:
- 'es': string (el mensaje en español)
- 'en': string (el mensaje en inglés)
";

    // 3. LLAMADA A LA API DE GOOGLE GEMINI
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . GEMINI_API_KEY;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'messages' => [
                        'type' => 'ARRAY',
                        'description' => 'Un array de 3 objetos, cada uno conteniendo un mensaje en español y otro en inglés.',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'es' => ['type' => 'STRING', 'description' => 'Mensaje de primer contacto en español.'],
                                'en' => ['type' => 'STRING', 'description' => 'Mensaje de primer contacto en inglés.']
                            ],
                            'required' => ['es', 'en']
                        ],
                        'minItems' => 3,
                        'maxItems' => 3
                    ]
                ],
                'required' => ['messages']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 4. PROCESAMIENTO Y VALIDACIÓN DE LA RESPUESTA
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502);
        error_log("Error de conexión/respuesta de IA en api-first-contact-message.php: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . $response_ai);
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA.', 'details' => $curl_error ?: 'No hay detalles de cURL.']);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado (partes de texto) en api-first-contact-message.php. Raw: " . $response_ai);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado (partes de texto).', 'raw_response' => $response_ai]);
        exit;
    }

    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['messages']) || !is_array($final_data['messages']) || count($final_data['messages']) !== 3) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido... Raw AI Text: " . trim($response_text) . ", JSON Error: " . json_last_error_msg());
        echo json_encode([
            'error' => 'La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta. Intenta de nuevo con tus datos.',
            'raw_ai_text' => trim($response_text),
            'json_last_error' => json_last_error_msg(),
            'decoded_data_debug' => $final_data
        ]);
        exit;
    }

    // 5. ENVÍO DE LA RESPUESTA FINAL
    ob_end_clean();
    echo json_encode($final_data, JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna en api-first-contact-message.php: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    while (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>