<?php
ob_start(); // Iniciar buffer de salida
require_once __DIR__ . '/../config.php'; // Incluye config.php para GEMINI_API_KEY

header('Content-Type: application/json');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y DATOS
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    // Verifica que la API Key esté definida y no vacía desde config.php
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada en el servidor. Verifica tu archivo config.php.']);
        exit;
    }

    $apiKey = GEMINI_API_KEY;

    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido. ' . json_last_error_msg()]);
        exit;
    }

    // --- Sanitización y Validación de campos ---
    $business_name = filter_var($data['business_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $phone = filter_var($data['phone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $website = filter_var($data['website'] ?? '', FILTER_SANITIZE_URL);
    $service = filter_var($data['service'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $zone = filter_var($data['zone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $client_summary = filter_var($data['client_summary'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS); // CAMBIO
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    
    // Validar que los campos obligatorios sanitizados no estén vacíos
    $required_fields_check = ['business_name', 'phone', 'service', 'zone', 'client_summary', 'value_proposition']; // CAMBIO
    foreach ($required_fields_check as $field) {
        if (empty(${$field})) {
            http_response_code(400);
            echo json_encode(['error' => "Falta el campo requerido: '{$field}'. Asegúrate de completar todos los datos."]);
            exit;
        }
    }

    // Limpiar la URL de 'http://' o 'https://' para un CTA más limpio
    $website_clean = $website;
    if (!empty($website_clean)) {
        $website_clean = preg_replace('/^^https?:\/\//', '', $website_clean);
    }
    
    // --- PROMPT MAESTRO (v3.2 - Consistencia y Estructura Robusta) ---
    $prompt = "Actúa como un copywriter experto en marketing de respuesta directa para el negocio local '{$business_name}'.
Tu tarea es generar 3 anuncios para Facebook, cada uno con un enfoque psicológico distinto.

--- CONTEXTO Y FILOSOFÍA DEL CURSO (Módulo 3: Creando el Anuncio Estratégico) ---
La estructura de un buen anuncio se basa en 3 pasos:
1.  **GANCHO:** La primera línea debe atraer la atención. Debe ser una pregunta o afirmación que toque el problema o deseo del cliente ideal para que se sienta identificado de inmediato. (Ej: '¿Llegas cansado del trabajo y aún tienes que limpiar?').
2.  **SOLUCIÓN:** Explica brevemente qué ofreces, cómo ayudas y en qué zona trabajas. (Ej: 'Hacemos limpieza profesional en [zona]. Tú descansas, nosotros dejamos tu casa impecable.').
3.  **LLAMADO A LA ACCIÓN (CTA):** Dile al cliente exactamente qué debe hacer para contactarte. Debe ser directo y fácil. (Ej: 'Escríbenos ahora y agenda tu servicio.').

--- TU TAREA ESPECÍFICA ---
Genera 3 anuncios para Facebook, cada uno con un enfoque distinto. **Debes inferir el problema y el deseo de forma clara a partir del resumen del cliente ideal:**
1.  **Anuncio Enfocado en el Problema:** El 'gancho' debe agitar el problema del cliente. La 'solucion' debe presentar el servicio como el alivio a ese problema.
2.  **Anuncio Enfocado en el Deseo:** El 'gancho' debe pintar una imagen del resultado ideal. La 'solucion' debe mostrar cómo el servicio entrega ese deseo.
3.  **Anuncio Enfocado en el Servicio:** El 'gancho' debe ser una presentación directa y profesional. La 'solucion' debe listar beneficios con checkmarks (✅).

REGLAS DE FORMATO Y PERSONALIZACIÓN:
- Usa el nombre del negocio '{$business_name}' en al menos uno de los anuncios.
- En la 'cta', SIEMPRE incluye el teléfono/WhatsApp: {$phone}.
- Si el usuario proporciona una página web ({$website_clean}), inclúyela también en la 'cta'.
- La respuesta final DEBE ser un objeto JSON válido con una clave 'anuncios', que es un array de 3 objetos. Cada objeto debe tener 'titulo', 'es' (con gancho, solucion, cta), y 'en' (con gancho, solucion, cta).
- Usa los títulos exactos en español: '💡 ANUNCIO ENFOCADO EN EL PROBLEMA', '💡 ANUNCIO ENFOCADO EN EL DESEO', '💡 ANUNCIO ENFOCADO EN EL SERVICIO'.
- **ASEGÚRATE de NO incluir 'http://' o 'https://' en las URLs en el CTA que generes.**

--- DATOS DEL ESTUDIANTE ---
- Nombre del Negocio: {$business_name}
- Teléfono/WhatsApp: {$phone}
- Página Web: {$website_clean}
- Servicio: {$service}
- Zona: {$zone}
- Resumen Cliente Ideal (Problema y Deseo Combinado): {$client_summary}"; // CAMBIO
    
    // --- ARQUITECTURA MEJORADA ---
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . GEMINI_API_KEY;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'anuncios' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'titulo' => ['type' => 'STRING'],
                                'es' => [
                                    'type' => 'OBJECT',
                                    'properties' => [
                                        'gancho' => ['type' => 'STRING'],
                                        'solucion' => ['type' => 'STRING'],
                                        'cta' => ['type' => 'STRING']
                                    ],
                                    'required' => ['gancho', 'solucion', 'cta']
                                ],
                                'en' => [
                                    'type' => 'OBJECT',
                                    'properties' => [
                                        'gancho' => ['type' => 'STRING'],
                                        'solucion' => ['type' => 'STRING'],
                                        'cta' => ['type' => 'STRING']
                                    ],
                                    'required' => ['gancho', 'solucion', 'cta']
                                ]
                            ],
                            'required' => ['titulo', 'es', 'en']
                        ],
                        'minItems' => 3,
                        'maxItems' => 3
                    ]
                ],
                'required' => ['anuncios']
            ]
        ]
    ];


    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90 
    ]);
    
    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 4. PROCESAMIENTO Y VALIDACIÓN DE LA RESPUESTA DE LA IA
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502);
        error_log("Error de conexión/respuesta de IA en api-ads-generator.php: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . $response_ai);
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA.', 'details' => $curl_error ?: 'No hay detalles de cURL.']);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado (partes de texto) en api-ads-generator.php. Raw: " . $response_ai);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado (parts of text).', 'raw_response' => $response_ai]);
        exit;
    }
    
    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['anuncios']) || !is_array($final_data['anuncios']) || count($final_data['anuncios']) !== 3) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta en api-ads-generator.php. Raw AI Text: " . trim($response_text) . ", JSON Error: " . json_last_error_msg());
        echo json_encode([
            'error' => 'La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta. Intenta de nuevo con tus datos.',
            'raw_ai_text' => trim($response_text),
            'json_error_message' => json_last_error_msg(),
            'decoded_data_debug' => $final_data
        ]);
        exit;
    }

    // 5. ENVÍO DE LA RESPUESTA FINAL
    ob_end_clean();
    echo json_encode($final_data);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna en api-ads-generator.php: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    while (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>